/******************************************************************************
 * © 2020 SAP SE or an SAP affiliate company. All rights reserved.            *
 ******************************************************************************/

package com.sap.cloud.mt.subscription;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.sap.cloud.mt.subscription.exceptions.InternalError;
import com.sap.cloud.mt.subscription.exceptions.ParameterError;
import com.sap.cloud.mt.subscription.json.Cloner;
import com.sap.cloud.mt.subscription.json.SubscriptionPayload;
import org.apache.commons.lang3.StringUtils;

import java.net.URL;
import java.time.Duration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.function.Supplier;

public class Tools {
    public static final String SECURE_CHARS = "^[_a-zA-Z0-9\\-]+$";
    private static final ObjectMapper objectMapper = new ObjectMapper();
    private static final String TENANT_SUCCESSFULLY_SUBSCRIBED_NO_APPLICATION_URL_PROVIDED = "tenant successfully subscribed - no application URL provided";
    public static final String ADDITIONAL_INFO_FOR_SM = "_";
    public static final String EVENT_TYPE = "eventType";

    private Tools() {
    }

    public static void checkExternalTenantId(String tenantId) throws ParameterError {
        if (!FilterTenants.realTenants().test(tenantId))
            throw new ParameterError("Tenant Id is identical to a technical container");
        Tools.checkTenantIdCharacters(tenantId);
    }

    public static void checkExternalTenantIds(List<String> tenantIds) throws ParameterError {
        for (String tenantId : tenantIds) {
            checkExternalTenantId(tenantId);
        }
    }

    public static void checkTenantIdCharacters(String tenantId) throws ParameterError {
        if (!tenantId.matches(SECURE_CHARS)) {
            throw new ParameterError("Tenant id contains forbidden characters");
        }
    }

    public static ObjectMapper getObjectMapper() {
        return objectMapper;
    }

    public static void waitSomeTime(Duration waitTime) {
        try {
            Thread.sleep(waitTime.toMillis());
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    }

    public static String getApplicationUrl(SubscriptionPayload subscriptionPayload, Supplier<URL> urlSupplier, Function<SubscriptionPayload, URL> urlFunction,
                                           String baseUiUrl, String urlSeparator) throws InternalError, ParameterError {
        if (subscriptionPayload == null) {
            return TENANT_SUCCESSFULLY_SUBSCRIBED_NO_APPLICATION_URL_PROVIDED;
        }
        var payloadAccess = SubscriptionPayloadAccess.create(subscriptionPayload.getMap());
        if (StringUtils.isBlank(payloadAccess.getSubDomain())) {
            return TENANT_SUCCESSFULLY_SUBSCRIBED_NO_APPLICATION_URL_PROVIDED;
        }
        //get application URL for CIS from exit
        URL urlFromExit = urlSupplier.get();
        if (urlFromExit == null) {
            urlFromExit = urlFunction.apply(Cloner.clone(subscriptionPayload));
        }
        String applicationUrl = urlFromExit != null ? urlFromExit.toExternalForm() : null;
        //build URL from properties if not set by exit
        if (applicationUrl == null)
            applicationUrl = UiUrlCreator.createUrl(payloadAccess.getSubDomain(), baseUiUrl, urlSeparator);
        if (StringUtils.isBlank(applicationUrl)) {
            applicationUrl = TENANT_SUCCESSFULLY_SUBSCRIBED_NO_APPLICATION_URL_PROVIDED;
        }
        return applicationUrl;
    }

    public static Map<String, Object> getSubscriptionPayloadForSidecar(SubscriptionPayload subscriptionPayload, ServiceCreateOptions serviceCreateOptions) {
        Map<String, Object> payload = new HashMap<>();
        payload.putAll(subscriptionPayload.getMap());
        if (StringUtils.isBlank((String) payload.get(EVENT_TYPE))) {
            payload.put(EVENT_TYPE, "CREATE");
        }
        boolean hasProvisioningParameters = false;
        boolean hasBindingParameters = false;
        if (serviceCreateOptions != null) {
            if (serviceCreateOptions.getProvisioningParameters() != null && !serviceCreateOptions.getProvisioningParameters().isEmpty()) {
                hasProvisioningParameters = true;
            }
            if (serviceCreateOptions.getBindingParameters() != null && !serviceCreateOptions.getBindingParameters().isEmpty()) {
                hasBindingParameters = true;
            }
        }
        if (hasProvisioningParameters || hasBindingParameters) {
            var application = new HashMap<String, Object>();
            payload.put("_application_", application);
            var sap = new HashMap<String, Object>();
            application.put("sap", sap);
            var serviceManager = new HashMap<String, Object>();
            sap.put("service-manager", serviceManager);
            if (hasProvisioningParameters) {
                serviceManager.put("provisioning_parameters", serviceCreateOptions.getProvisioningParameters());
            }
            if (hasBindingParameters) {
                serviceManager.put("binding_parameters", serviceCreateOptions.getBindingParameters());
            }
        }
        return payload;
    }

    /*
      New payload format under "_":
      "hdi": {
        "create": {
                    "database_id": "<ID>"
        },
        "bind": {
                     "key": "value"
        }
      }           */
    public static Map<String, Object> getProvisioningServicePayload(SubscriptionPayload subscriptionPayload, ServiceCreateOptions serviceCreateOptions) {
        Map<String, Object> payload = new HashMap<>();
        payload.putAll(subscriptionPayload.getMap());
        boolean hasProvisioningParameters = false;
        boolean hasBindingParameters = false;
        if (serviceCreateOptions != null) {
            if (serviceCreateOptions.getProvisioningParameters() != null && !serviceCreateOptions.getProvisioningParameters().isEmpty()) {
                hasProvisioningParameters = true;
            }
            if (serviceCreateOptions.getBindingParameters() != null && !serviceCreateOptions.getBindingParameters().isEmpty()) {
                hasBindingParameters = true;
            }
        }
        if (hasProvisioningParameters || hasBindingParameters) {
            var hdi = new HashMap<String, Object>();
            var additionalInfoForSM = new HashMap<String, Object>();
            payload.put(ADDITIONAL_INFO_FOR_SM, additionalInfoForSM);
            additionalInfoForSM.put("hdi", hdi);
            if (hasProvisioningParameters) {
                hdi.put("create", serviceCreateOptions.getProvisioningParameters());
            }
            if (hasBindingParameters) {
                hdi.put("bind", serviceCreateOptions.getBindingParameters());
            }
        }
        return payload;
    }

    /**
     * Lazy json creation for log output.
     *
     * @param callable A supplier that returns an object when called that is then converted into Json
     * @return An object with an overwritten toString method that returns a Json
     */
    public static Object lazyJson(Supplier<Object> callable) {
        return new Object() {
            @Override
            public String toString() {
                if (callable == null) {
                    return """
                            {
                              "value": "null"
                            }
                            """;
                }
                try {
                    Object result = callable.get();
                    if (result == null) {
                        return "";
                    }
                    return objectMapper.writeValueAsString(result);
                } catch (Exception e) {
                    return "";
                }
            }
        };
    }
}