/*
 * ----------------------------------------------------------------
 * © 2021 SAP SE or an SAP affiliate company. All rights reserved.
 * ----------------------------------------------------------------
 *
 */

package com.sap.cloud.mt.tools.api;

import com.sap.cloud.mt.tools.api.ResponseChecker.CheckResult;
import com.sap.cloud.mt.tools.api.ServiceCallMediators.FromCreate;
import com.sap.cloud.mt.tools.api.ServiceEndpointMediators.End;
import com.sap.cloud.mt.tools.api.ServiceEndpointMediators.FromAt;
import com.sap.cloud.mt.tools.api.ServiceEndpointMediators.FromAuthTokenRetry;
import com.sap.cloud.mt.tools.api.ServiceEndpointMediators.FromCodeChecker;
import com.sap.cloud.mt.tools.api.ServiceEndpointMediators.FromConfig;
import com.sap.cloud.mt.tools.api.ServiceEndpointMediators.FromDestination;
import com.sap.cloud.mt.tools.api.ServiceEndpointMediators.FromDoNotRetry;
import com.sap.cloud.mt.tools.api.ServiceEndpointMediators.FromPath;
import com.sap.cloud.mt.tools.api.ServiceEndpointMediators.FromRetry;
import com.sap.cloud.mt.tools.api.ServiceEndpointMediators.FromReturnCodesOrExceptions;
import com.sap.cloud.mt.tools.api.ServiceEndpointMediators.RetryAuthenticationTokenDetermination;
import com.sap.cloud.mt.tools.exception.InternalException;
import com.sap.cloud.mt.tools.impl.ServiceCallBuilder;
import org.apache.commons.lang3.StringUtils;

import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.IntFunction;

/**
 * Caller of a specified http endpoint with a specified htp method
 */
public class ServiceEndpoint {
    private final String destinationName;
    // base URL of the service
    private final String baseUrl;
    //the path added to the service URl
    private final String path;
    private final ResilienceConfig resilienceConfig;
    private final ResilienceConfig resilienceConfigAuth;
    private final ResponseChecker responseChecker;
    private final Set<Class<? extends Throwable>> authenticationExceptionsForRetry = new HashSet<>();


    private ServiceEndpoint(String destinationName, String baseUrl, String path, ResilienceConfig resilienceConfig,
                            ResponseChecker responseChecker, ResilienceConfig resilienceConfigAuth,
                            Set<Class<? extends Throwable>> authenticationExceptionsForRetry) {
        this.destinationName = destinationName;
        this.baseUrl = baseUrl;
        this.path = path;
        this.resilienceConfig = resilienceConfig;
        this.responseChecker = responseChecker;
        this.resilienceConfigAuth = resilienceConfigAuth;
        this.authenticationExceptionsForRetry.addAll(authenticationExceptionsForRetry);
    }

    public static ServiceEndpointMediators.FromCreate create() {
        return ServiceEndpointBuilder.createBuilder();
    }

    public String getDestinationName() {
        return destinationName;
    }

    public String getBaseUrl() {
        return baseUrl;
    }

    public String getPath() {
        return path;
    }

    public ResilienceConfig getResilienceConfig() {
        return resilienceConfig;
    }

    public ResilienceConfig getResilienceConfigAuth() {
        return resilienceConfigAuth;
    }

    public ResponseChecker getResponseChecker() {
        return responseChecker;
    }

    public Set<Class<? extends Throwable>> getAuthenticationExceptionsForRetry() {
        return authenticationExceptionsForRetry;
    }

    public FromCreate createServiceCall() {
        return ServiceCallBuilder.createBuilder(this);
    }

    public static class ServiceEndpointBuilder implements ServiceEndpointMediators.FromCreate, FromAt, FromDestination, FromPath,
            FromCodeChecker, FromRetry, FromReturnCodesOrExceptions, FromAuthTokenRetry, FromConfig,
            FromDoNotRetry, RetryAuthenticationTokenDetermination, End {
        private Optional<String> destinationName = Optional.empty();
        // base URL of the service
        private Optional<String> baseUrl = Optional.empty();
        //the path added to the service URl
        private Optional<String> path = Optional.empty();
        private ResilienceConfig resilienceConfigService = ResilienceConfig.NONE;
        private ResilienceConfig resilienceConfigAuth = ResilienceConfig.NONE;
        private Optional<IntFunction<Exception>> checkFunction = Optional.empty();
        private Set<Integer> returnCodesForRetry = new HashSet<>();
        private Set<Class<? extends Throwable>> authenticationExceptionsForRetry = new HashSet<>();
        private RetryMode retryMode;

        private ServiceEndpointBuilder() {
        }

        public static ServiceEndpointMediators.FromCreate createBuilder() {
            return new ServiceEndpointBuilder();
        }

        @Override
        public FromAt at(String baseUrl) {
            this.baseUrl = Optional.ofNullable(baseUrl);
            return this;
        }

        @Override
        public FromDestination destinationName(String destinationName) {
            this.destinationName = Optional.ofNullable(destinationName);
            return this;
        }

        @Override
        public FromPath path(String path) {
            this.path = Optional.ofNullable(path);
            return this;
        }

        @Override
        public FromCodeChecker returnCodeChecker(IntFunction<Exception> checkFunction) {
            this.checkFunction = Optional.ofNullable(checkFunction);
            return this;
        }

        @Override
        public FromCodeChecker throwNoExceptionForReturnCode() {
            return this;
        }

        @Override
        public FromRetry retry() {
            retryMode = RetryMode.SERVICE;
            return this;
        }

        @Override
        public FromAuthTokenRetry retryAuthenticationTokenDetermination() {
            retryMode = RetryMode.AUTHENTICATION;
            return this;
        }

        @Override
        public FromReturnCodesOrExceptions forExceptions(Set<Class<? extends Throwable>> exceptions) {
            if (exceptions != null) {
                this.authenticationExceptionsForRetry.addAll(exceptions);
            }
            return this;
        }

        @Override
        public FromReturnCodesOrExceptions forExceptions(List<Class<? extends Throwable>> exceptions) {
            if (exceptions != null) {
                this.authenticationExceptionsForRetry.addAll(exceptions);
            }
            return this;
        }

        @Override
        public FromReturnCodesOrExceptions forExceptions(Class<? extends Throwable>... exceptions) {
            Arrays.stream(exceptions).sequential().forEach(this.authenticationExceptionsForRetry::add);
            return this;
        }

        @Override
        public FromDoNotRetry doNotRetry() {
            return this;
        }

        @Override
        public FromReturnCodesOrExceptions forReturnCodes(Set<Integer> returnCodes) {
            if (returnCodes != null) {
                this.returnCodesForRetry.addAll(returnCodes);
            }
            return this;
        }

        @Override
        public FromReturnCodesOrExceptions forReturnCodes(List<Integer> returnCodes) {
            if (returnCodes != null) {
                this.returnCodesForRetry.addAll(returnCodes);
            }
            return this;
        }

        @Override
        public FromReturnCodesOrExceptions forReturnCodes(Integer... returnCodes) {
            Arrays.stream(returnCodes).sequential().forEach(this.returnCodesForRetry::add);
            return this;
        }

        @Override
        public FromConfig config(ResilienceConfig resilienceConfig) {
            if (resilienceConfig != null) {
                if (retryMode == RetryMode.SERVICE) {
                    this.resilienceConfigService = resilienceConfig;
                } else {
                    this.resilienceConfigAuth = resilienceConfig;
                }
            }
            return this;
        }

        @Override
        public ServiceEndpoint end() throws InternalException {
            checkParameters();
            ResponseChecker responseChecker;
            if (!checkFunction.isPresent() && returnCodesForRetry.isEmpty()) {
                responseChecker = ResponseChecker.NONE;
            } else {
                if (!checkFunction.isPresent()) {
                    checkFunction = Optional.of(code -> null);
                }
                responseChecker = createResponseChecker(checkFunction.get(), returnCodesForRetry);
            }
            return new ServiceEndpoint(destinationName.get(), baseUrl.get(), path.get(), resilienceConfigService, responseChecker,
                    resilienceConfigAuth, authenticationExceptionsForRetry);
        }

        private void checkParameters() throws InternalException {
            if (!destinationName.isPresent() || StringUtils.isEmpty(destinationName.get())) {
                throw new InternalException("No service destination name set");
            }
            if (!baseUrl.isPresent() || StringUtils.isEmpty(baseUrl.get())) {
                throw new InternalException("No service base URL set");
            }
            if (!path.isPresent() || StringUtils.isEmpty(path.get())) {
                throw new InternalException("No service path set");
            }
        }

        private static ResponseChecker createResponseChecker(IntFunction<Exception> checkFunction, Set<Integer> returnCodesForRetry) {
            return code -> {
                Exception exception = checkFunction.apply(code);
                if (returnCodesForRetry.contains(code)) {
                    return new CheckResult(exception, true);
                }
                if (exception != null) {
                    return new CheckResult(exception, false);
                }
                return CheckResult.NO_PROBLEM;
            };
        }

        private enum RetryMode {
            SERVICE, AUTHENTICATION;
        }
    }

}
