/*
 * ----------------------------------------------------------------
 * © 2019-2021 SAP SE or an SAP affiliate company. All rights reserved.
 * ----------------------------------------------------------------
 *
 */

package com.sap.cloud.mt.tools.impl;

import com.sap.cloud.mt.tools.api.HttpMethod;
import com.sap.cloud.mt.tools.api.QueryParameter;
import com.sap.cloud.mt.tools.api.ServiceCall;
import com.sap.cloud.mt.tools.api.ServiceCallMediators.FromAuthenticate;
import com.sap.cloud.mt.tools.api.ServiceCallMediators.FromCreate;
import com.sap.cloud.mt.tools.api.ServiceCallMediators.FromHttp;
import com.sap.cloud.mt.tools.api.ServiceCallMediators.FromHttpMethod;
import com.sap.cloud.mt.tools.api.ServiceCallMediators.FromQuery;
import com.sap.cloud.mt.tools.api.ServiceEndpoint;
import com.sap.cloud.mt.tools.exception.InternalException;
import org.apache.commons.lang3.StringUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import static com.sap.cloud.mt.tools.api.ServiceCallMediators.End;
import static com.sap.cloud.mt.tools.api.ServiceCallMediators.FromPathParameter;
import static com.sap.cloud.mt.tools.api.ServiceCallMediators.FromPayload;

/**
 * Use this builder to create an instance of the service abstraction.
 */
public final class ServiceCallBuilder implements FromCreate, FromHttp, FromHttpMethod, FromPayload, FromPathParameter, FromQuery, FromAuthenticate, End {
    private Optional<ServiceEndpoint> serviceEndpoint = Optional.empty();
    private final Map<String, String> headerFields = new HashMap<>();
    private final List<QueryParameter> queryParameters = new ArrayList<>();
    private Optional<SupplierWithException<String>> authenticationTokenSupplier = Optional.empty();
    private Optional<?> payload = Optional.empty();
    private Optional<String> pathParameter = Optional.empty();
    private Optional<String> authenticationToken = Optional.empty();
    private HttpMethod httpMethod;

    private ServiceCallBuilder(ServiceEndpoint serviceEndpoint) {
        this.serviceEndpoint = Optional.ofNullable(serviceEndpoint);
    }

    @Override
    public FromHttp http() {
        return this;
    }

    @Override
    public FromHttpMethod get() {
        this.httpMethod = HttpMethod.GET;
        return this;
    }

    @Override
    public FromHttpMethod put() {
        this.httpMethod = HttpMethod.PUT;
        return this;
    }

    @Override
    public FromHttpMethod post() {
        this.httpMethod = HttpMethod.POST;
        return this;
    }

    @Override
    public FromHttpMethod delete() {
        this.httpMethod = HttpMethod.DELETE;
        return this;
    }

    @Override
    public FromHttpMethod patch() {
        this.httpMethod = HttpMethod.PATCH;
        return this;
    }

    @Override
    public <P> FromPayload payload(P payload) {
        this.payload = Optional.ofNullable(payload);
        return this;
    }

    @Override
    public FromPayload withoutPayload() {
        return this;
    }

    @Override
    public FromPathParameter pathParameter(String pathParameter) {
        this.pathParameter = Optional.ofNullable(pathParameter);
        return this;
    }

    @Override
    public FromPathParameter noPathParameter() {
        return this;
    }

    @Override
    public FromQuery query(List<QueryParameter> queryParameters) {
        if (queryParameters != null) {
            this.queryParameters.addAll(queryParameters);
        }
        return this;
    }

    @Override
    public FromQuery noQuery() {
        return this;
    }

    @Override
    public FromAuthenticate authenticationTokenSupplier(SupplierWithException<String> authenticationTokenSupplier) {
        this.authenticationTokenSupplier = Optional.ofNullable(authenticationTokenSupplier);
        return this;
    }

    @Override
    public FromAuthenticate authenticationToken(String authenticationToken) {
        this.authenticationToken = Optional.ofNullable(authenticationToken);
        return this;
    }

    @Override
    public FromAuthenticate withoutAuthentication() {
        return this;
    }

    @Override
    public End insertHeaderFields(Map<String, String> headerFields) {
        this.headerFields.putAll(headerFields);
        return this;
    }

    public static FromCreate createBuilder(ServiceEndpoint serviceEndpoint) {
        return new ServiceCallBuilder(serviceEndpoint);
    }

    @Override
    public ServiceCall end() throws InternalException {
        if (!serviceEndpoint.isPresent()) {
            throw new InternalException("No service endpoint specified");
        }
        if (pathParameter.isPresent() && StringUtils.isEmpty(pathParameter.get())) {
            throw new InternalException("An empty path parameter was specified");
        }
        if (authenticationToken.isPresent() && StringUtils.isEmpty(authenticationToken.get())) {
            throw new InternalException("An empty authentication token was specified");
        }
        return new ServiceCallImpl(httpMethod, serviceEndpoint.get(), pathParameter, payload, authenticationTokenSupplier, headerFields, authenticationToken,
                queryParameters);
    }
}
