/*
 * ----------------------------------------------------------------
 * © 2021 SAP SE or an SAP affiliate company. All rights reserved.
 * ----------------------------------------------------------------
 *
 */

package com.sap.cloud.mt.tools.impl;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.sap.cloud.mt.tools.api.HttpMethod;
import com.sap.cloud.mt.tools.api.QueryParameter;
import com.sap.cloud.mt.tools.api.ResponseChecker;
import com.sap.cloud.mt.tools.api.ServiceCall;
import com.sap.cloud.mt.tools.api.ServiceEndpoint;
import com.sap.cloud.mt.tools.api.ServiceResponse;
import com.sap.cloud.mt.tools.exception.ServiceException;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpClientAccessor;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestination;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHeaders;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.methods.RequestBuilder;
import org.apache.http.entity.StringEntity;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import static org.apache.http.HttpHeaders.CONTENT_TYPE;

public class ServiceCallImpl implements ServiceCall {
    private static final Logger logger = LoggerFactory.getLogger(ServiceCallImpl.class);
    private static final String APPLICATION_JSON = "application/json";
    private static final String SLASH = "/";
    private final ServiceEndpoint serviceEndpoint;
    private final Optional<String> pathParameter;
    private final Map<String, String> headerFields = new HashMap<>();
    private final List<QueryParameter> queryParameters = new ArrayList<>();
    private final Optional<?> payload;
    private final Optional<SupplierWithException<String>> authenticationTokenSupplier;
    private final Optional<String> authenticationToken;
    private final HttpMethod httpMethod;
    private final ObjectMapper objectMapper;

    protected ServiceCallImpl(HttpMethod httpMethod, ServiceEndpoint serviceEndpoint, Optional<String> pathParameter, Optional<?> payload,
                              Optional<SupplierWithException<String>> authenticationTokenSupplier, Map<String, String> headerFields,
                              Optional<String> authenticationToken, List<QueryParameter> queryParameters) {
        this.httpMethod = httpMethod;
        this.serviceEndpoint = serviceEndpoint;
        this.pathParameter = pathParameter;
        this.payload = payload;
        this.headerFields.putAll(headerFields);
        this.queryParameters.addAll(queryParameters);
        if (authenticationToken.isPresent() && authenticationTokenSupplier.isPresent()) {
            throw new IllegalArgumentException("It isn't allowed to specify an authentication token together with an authentication token supplier");
        }
        this.authenticationTokenSupplier = authenticationTokenSupplier;
        this.authenticationToken = authenticationToken;
        this.objectMapper = new ObjectMapper();
        this.objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
    }

    @Override
    public <R> ServiceResponse<R> execute(Class<R> responseType) throws ServiceException {
        return execute(httpMethod, responseType);
    }

    @Override
    public ServiceResponse<Void> execute() throws ServiceException {
        return execute(httpMethod, Void.class);
    }

    private <R> ServiceResponse<R> execute(HttpMethod method, Class<R> responseType) throws ServiceException {
        RequestBuilder requestBuilder = getRequestBuilder(method, serviceEndpoint.getPath(), pathParameter)
                .setHeader(CONTENT_TYPE, APPLICATION_JSON);
        queryParameters.forEach(queryParameter -> requestBuilder.addParameter(queryParameter.getName(), queryParameter.getValue()));
        if (payload.isPresent()) {
            requestBuilder.setEntity(asEntity(payload.get()));
        }
        HttpUriRequest request = requestBuilder.build();
        setAuthentication(request);
        headerFields.forEach(request::setHeader);
        Retry retryServiceCall = Retry.RetryBuilder.create().retryExceptions(RetryException.class)
                .numOfRetries(serviceEndpoint.getResilienceConfig().getNumOfRetries())
                .waitTime(serviceEndpoint.getResilienceConfig().getRetryInterval()).build();
        try {
            return retryServiceCall.execute(() -> execute(request, responseType, serviceEndpoint.getResponseChecker()));
        } catch (RetryException retryException) {
            if (retryException.getCause() == null) {
                //checker didn't throw an error, only requested a retry =>normal exit
                return (ServiceResponse<R>) retryException.getResponse();
            } else {
                throw new ServiceException(retryException.getCause(), retryException.getResponse());
            }
        } catch (Exception e) {
            // no-retry is always triggered by exception => cause is filled
            NoRetryException noRetryException = (NoRetryException) e;
            throw new ServiceException(noRetryException.getCause(), noRetryException.getResponse());
        }
    }

    private void setAuthentication(HttpUriRequest request) throws ServiceException {
        try {
            if (authenticationToken.isPresent()) {
                request.addHeader(HttpHeaders.AUTHORIZATION, authenticationToken.get());
            } else if (authenticationTokenSupplier.isPresent()) {
                Retry.RetryBuilder.create().retryExceptions(serviceEndpoint.getAuthenticationExceptionsForRetry())
                        .numOfRetries(serviceEndpoint.getResilienceConfigAuth().getNumOfRetries())
                        .waitTime(serviceEndpoint.getResilienceConfigAuth().getRetryInterval()).build()
                        .execute(() -> request.addHeader(HttpHeaders.AUTHORIZATION, authenticationTokenSupplier.get().get()));
            }
        } catch (Exception e) {
            throw new ServiceException(e, null);
        }
    }

    private RequestBuilder getRequestBuilder(HttpMethod method, String path, Optional<String> pathParameter) throws ServiceException {
        switch (method) {
            case GET: {
                return RequestBuilder.get(getTotalPath(path, pathParameter));
            }
            case PUT: {
                return RequestBuilder.put(getTotalPath(path, pathParameter));
            }
            case POST: {
                return RequestBuilder.post(getTotalPath(path, pathParameter));
            }
            case DELETE: {
                return RequestBuilder.delete(getTotalPath(path, pathParameter));
            }
            case PATCH: {
                return RequestBuilder.patch(getTotalPath(path, pathParameter));
            }
            default: {
                throw new ServiceException("Http method " + method.name() + " not known", null);
            }
        }
    }

    private String getTotalPath(String path, Optional<String> pathParameter) {
        StringBuilder pathBuilder = new StringBuilder(path);
        if (pathParameter.isPresent() && StringUtils.isNotEmpty(pathParameter.get())) {
            //get rid of leading slash
            String adjustedPathParameter = pathParameter.get();
            if (pathParameter.get().startsWith(SLASH)) {
                if (pathParameter.get().length() == 1) {
                    adjustedPathParameter = "";
                } else {
                    adjustedPathParameter = pathParameter.get().substring(1);
                }
            }
            if (!path.endsWith(SLASH)) {
                pathBuilder.append(SLASH);
            }
            pathBuilder.append(adjustedPathParameter);
        }
        return pathBuilder.toString();
    }

    private <R> ServiceResponse<R> execute(HttpUriRequest request, Class<R> responseType, ResponseChecker responseChecker) throws RetryException, NoRetryException {
        ServiceResponse<R> serviceResponse = new ServiceResponse<>();
        try {
            HttpDestination destination = ServiceDestinations.getDestination(serviceEndpoint.getDestinationName(), serviceEndpoint.getBaseUrl());
            HttpClient client = HttpClientAccessor.getHttpClient(destination);
            logger.debug("Service {} is called with url {} and path {}", serviceEndpoint.getDestinationName(), destination.getUri(), request.getURI());
            try (CloseableHttpResponse response = (CloseableHttpResponse) client.execute(request)) {
                int responseCode = response.getStatusLine().getStatusCode();
                serviceResponse.setHttpStatusCode(responseCode);
                serviceResponse.setHeaders(response.getAllHeaders());
                String returnedPayloadAsString = "";
                if (response.getEntity() != null && responseType != Void.class) {
                    returnedPayloadAsString = EntityUtils.toString(response.getEntity());
                    serviceResponse.setPayload(Optional.ofNullable(asResultType(returnedPayloadAsString, responseType)));
                }
                if (response.containsHeader(HttpHeaders.ETAG)) {
                    serviceResponse.setETag(Optional.of(response.getLastHeader(HttpHeaders.ETAG).getValue()));
                }
                ResponseChecker.CheckResult checkResult = responseChecker.checkCode(responseCode);
                Optional<Exception> exception = checkResult.getException();
                if (checkResult.isTemporaryProblem()) {
                    logger.debug("A temporary problem was detected. Service call returned with code {}", responseCode);
                    if (StringUtils.isNotEmpty(returnedPayloadAsString)) {
                        logger.debug("Service {} was called with url {} and path {} and returned payload {}", serviceEndpoint.getDestinationName(), destination.getUri(), request.getURI(), returnedPayloadAsString);
                    }
                    if (exception.isPresent()) {
                        throw new RetryException(exception.get(), serviceResponse);
                    }
                    // request a retry but the caller doesn't demand an exception
                    throw new RetryException(serviceResponse);
                }
                // an exception without retry
                if (exception.isPresent()) {
                    logger.error("A non temporary problem was detected. Service call returned with code {}", responseCode);
                    if (StringUtils.isNotEmpty(returnedPayloadAsString)) {
                        logger.debug("Service {} was called with url {} and path {} and returned payload {}", serviceEndpoint.getDestinationName(), destination.getUri(), request.getURI(), returnedPayloadAsString);
                    }
                    throw new NoRetryException(exception.get(), serviceResponse);
                }
            }
        } catch (RetryException | NoRetryException exception) {
            throw exception;
        } catch (IOException ioException) {
            throw new RetryException(ioException, serviceResponse);
        } catch (Exception e) {
            throw new NoRetryException(e, serviceResponse);
        }
        return serviceResponse;
    }

    private <R> R asResultType(String payload, Class<R> responseType) {
        if (StringUtils.isEmpty(payload)) {
            payload = "{}";
        }
        if (responseType == String.class) {
            return (R) payload;
        }
        try {
            return objectMapper.readValue(payload, responseType);
        } catch (JsonProcessingException e) {
            return null;
        }
    }

    private HttpEntity asEntity(Object payload) throws ServiceException {

        try {
            if (payload instanceof String) {
                return new StringEntity((String) payload);
            } else {
                return new StringEntity(new ObjectMapper().writerWithDefaultPrettyPrinter().writeValueAsString(payload));
            }
        } catch (JsonProcessingException | UnsupportedEncodingException e) {
            throw new ServiceException(e, null);
        }
    }
}