/*
 * ----------------------------------------------------------------
 * © 2021 SAP SE or an SAP affiliate company. All rights reserved.
 * ----------------------------------------------------------------
 *
 */

package com.sap.cloud.mt.tools.api;

import com.sap.cloud.mt.tools.api.ResponseChecker.CheckResult;
import com.sap.cloud.mt.tools.api.ServiceCallMediators.FromCreate;
import com.sap.cloud.mt.tools.api.ServiceEndpointMediators.End;
import com.sap.cloud.mt.tools.api.ServiceEndpointMediators.FromCodeChecker;
import com.sap.cloud.mt.tools.api.ServiceEndpointMediators.FromConfig;
import com.sap.cloud.mt.tools.api.ServiceEndpointMediators.FromDestinationName;
import com.sap.cloud.mt.tools.api.ServiceEndpointMediators.FromDoNotRetry;
import com.sap.cloud.mt.tools.api.ServiceEndpointMediators.FromPath;
import com.sap.cloud.mt.tools.api.ServiceEndpointMediators.FromRetry;
import com.sap.cloud.mt.tools.api.ServiceEndpointMediators.FromReturnCodes;
import com.sap.cloud.mt.tools.exception.InternalException;
import com.sap.cloud.mt.tools.impl.ServiceCallBuilder;
import org.apache.commons.lang3.StringUtils;

import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.IntFunction;

/**
 * Caller of a specified http endpoint with a specified htp method
 */
public class ServiceEndpoint {
    private final String destinationName;
    //the path added to the service URl
    private final String path;
    private final ResilienceConfig resilienceConfig;
    private final ResponseChecker responseChecker;

    private ServiceEndpoint(String destinationName, String path, ResilienceConfig resilienceConfig, ResponseChecker responseChecker) {
        this.destinationName = destinationName;
        this.path = path;
        this.resilienceConfig = resilienceConfig;
        this.responseChecker = responseChecker;
    }

    public static ServiceEndpointMediators.FromCreate create() {
        return ServiceEndpointBuilder.createBuilder();
    }

    public String getDestinationName() {
        return destinationName;
    }

    public String getPath() {
        return path;
    }

    public ResilienceConfig getResilienceConfig() {
        return resilienceConfig;
    }


    public ResponseChecker getResponseChecker() {
        return responseChecker;
    }

    public FromCreate createServiceCall() {
        return ServiceCallBuilder.createBuilder(this);
    }

    public static class ServiceEndpointBuilder implements ServiceEndpointMediators.FromCreate, FromDestinationName, FromPath,
            FromCodeChecker, FromRetry, FromReturnCodes, FromConfig,
            FromDoNotRetry, End {
        // Destination of the service
        private Optional<String> destinationName = Optional.empty();
        //the path added to the service URl
        private Optional<String> path = Optional.empty();
        private ResilienceConfig resilienceConfigService = ResilienceConfig.NONE;
        private Optional<IntFunction<Exception>> checkFunction = Optional.empty();
        private Set<Integer> returnCodesForRetry = new HashSet<>();

        private ServiceEndpointBuilder() {
        }

        public static ServiceEndpointMediators.FromCreate createBuilder() {
            return new ServiceEndpointBuilder();
        }

        private static ResponseChecker createResponseChecker(IntFunction<Exception> checkFunction, Set<Integer> returnCodesForRetry) {
            return code -> {
                Exception exception = checkFunction.apply(code);
                if (returnCodesForRetry.contains(code)) {
                    return new CheckResult(exception, true);
                }
                if (exception != null) {
                    return new CheckResult(exception, false);
                }
                return CheckResult.NO_PROBLEM;
            };
        }

        @Override
        public FromDestinationName destinationName(String destinationName) {
            this.destinationName = Optional.ofNullable(destinationName);
            return this;
        }

        @Override
        public FromPath path(String path) {
            this.path = Optional.ofNullable(path);
            return this;
        }

        @Override
        public FromCodeChecker returnCodeChecker(IntFunction<Exception> checkFunction) {
            this.checkFunction = Optional.ofNullable(checkFunction);
            return this;
        }

        @Override
        public FromCodeChecker throwNoExceptionForReturnCode() {
            return this;
        }

        @Override
        public FromRetry retry() {
            return this;
        }

        @Override
        public FromDoNotRetry doNotRetry() {
            return this;
        }

        @Override
        public FromReturnCodes forReturnCodes(Set<Integer> returnCodes) {
            if (returnCodes != null) {
                this.returnCodesForRetry.addAll(returnCodes);
            }
            return this;
        }

        @Override
        public FromReturnCodes forReturnCodes(List<Integer> returnCodes) {
            if (returnCodes != null) {
                this.returnCodesForRetry.addAll(returnCodes);
            }
            return this;
        }

        @Override
        public FromReturnCodes forReturnCodes(Integer... returnCodes) {
            Arrays.stream(returnCodes).sequential().forEach(this.returnCodesForRetry::add);
            return this;
        }

        @Override
        public FromConfig config(ResilienceConfig resilienceConfig) {
            if (resilienceConfig != null) {
                this.resilienceConfigService = resilienceConfig;
            }
            return this;
        }

        @Override
        public ServiceEndpoint end() throws InternalException {
            checkParameters();
            ResponseChecker responseChecker;
            if (!checkFunction.isPresent() && returnCodesForRetry.isEmpty()) {
                responseChecker = ResponseChecker.NONE;
            } else {
                if (!checkFunction.isPresent()) {
                    checkFunction = Optional.of(code -> null);
                }
                responseChecker = createResponseChecker(checkFunction.get(), returnCodesForRetry);
            }
            return new ServiceEndpoint(destinationName.get(), path.get(), resilienceConfigService, responseChecker);
        }

        private void checkParameters() throws InternalException {
            if (!destinationName.isPresent() || StringUtils.isEmpty(destinationName.get())) {
                throw new InternalException("No service destination name set");
            }
            if (!path.isPresent() || StringUtils.isEmpty(path.get())) {
                throw new InternalException("No service path set");
            }
        }
    }
}
