/*
 * ----------------------------------------------------------------
 * © 2021 SAP SE or an SAP affiliate company. All rights reserved.
 * ----------------------------------------------------------------
 *
 */

package com.sap.cloud.mt.tools.impl;


import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Set;

public class Retry {
    private final int numOfRetries;
    private final List<Class> retryExceptions = new ArrayList<>();//NOSONAR
    private final Duration waitTime;

    private Retry(int numOfRetries, List<Class<? extends Throwable>> retryExceptions, Duration waitTime) {//NOSONAR
        this.numOfRetries = numOfRetries;
        this.retryExceptions.addAll(retryExceptions);
        if (waitTime != null) {
            this.waitTime = waitTime;
        } else {
            this.waitTime = Duration.ofSeconds(1);
        }
    }

    public <T> T execute(SupplierWithException<T> supplier) throws Exception {
        int retryRun = -1;
        do {
            retryRun++;
            try {
                return supplier.get();
            } catch (Throwable e) {//NOSONAR
                if (retryRun < numOfRetries && isARetryException(e)) {
                    sleep();
                } else {
                    throw e;
                }
            }
        } while (true);
    }

    public void execute(VoidConsumerWithException consumer) throws Exception {
        int retryRun = -1;
        do {
            retryRun++;
            try {
                consumer.accept();
                return;
            } catch (Throwable e) {//NOSONAR
                if (retryRun < numOfRetries && isARetryException(e)) {
                    sleep();
                } else {
                    throw e;
                }
            }
        } while (true);
    }

    private void sleep() {
        try {
            Thread.sleep(waitTime.toMillis());
        } catch (InterruptedException interruptedException) {
            Thread.currentThread().interrupt();
        }
    }

    private boolean isARetryException(Throwable e) {
        return retryExceptions.stream().anyMatch(clazz -> clazz.isInstance(e));
    }

    public static final class RetryBuilder {
        private int numOfRetries;
        private List<Class<? extends Throwable>> retryExceptions = new ArrayList<>();//NOSONAR
        private Duration waitTime;

        private RetryBuilder() {
        }

        public static RetryBuilder create() { //NOSONAR
            return new RetryBuilder();
        }

        public RetryBuilder numOfRetries(int numOfRetries) {
            this.numOfRetries = numOfRetries;
            return this;
        }

        public RetryBuilder retryExceptions(Class<? extends Throwable>... retryExceptions) {
            this.retryExceptions = Arrays.asList(retryExceptions);
            return this;
        }

        public RetryBuilder retryExceptions(Set<Class<? extends Throwable>> retryExceptions) {
            if (retryExceptions != null) {
                this.retryExceptions.addAll(retryExceptions);
            }
            return this;
        }

        public RetryBuilder waitTime(Duration waitTime) {
            this.waitTime = waitTime;
            return this;
        }

        public Retry build() {
            return new Retry(numOfRetries, retryExceptions, waitTime);
        }
    }
}
