/*
 * Decompiled with CFR 0.152.
 */
package com.sap.cloud.crypto.keystore.service;

import com.sap.cloud.crypto.keystore.api.KeyStoreServiceException;
import com.sap.cloud.crypto.keystore.service.KeyStoreProvider;
import com.sap.cloud.crypto.keystore.service.StorageLevel;
import com.sap.cloud.crypto.keystore.service.types.TypesMapper;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.SecureRandom;
import java.security.cert.CertificateException;
import java.util.Arrays;
import java.util.Random;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KeyStoreData {
    private static final Logger LOG = LoggerFactory.getLogger(KeyStoreData.class);
    private static Random RANDOM;
    private static final int PREFIX_LENGTH = 64;
    private String name;
    private String keyStoreTypeProvidedByStorage;
    private KeyStore keyStore;
    private long versionStamp;
    private byte[] keystoreContent;
    private byte[] correctPasswordDigest;
    private byte[] passwordPrefixBytes;
    private boolean isKeyStoreLoaded;
    private StorageLevel storageLevel;

    static {
        try {
            RANDOM = SecureRandom.getInstance("SHA1PRNG");
        }
        catch (NoSuchAlgorithmException e) {
            LOG.error("Failed to retrieve a secure random generator. Falling back to a simple random number generator.", (Throwable)e);
            RANDOM = new SecureRandom();
        }
    }

    public KeyStoreData(String keyStoreName, String keyStoreTypeProvidedByStorage, long versionStamp, byte[] keystoreContent, StorageLevel storageLevel) throws KeyStoreServiceException {
        this.name = keyStoreName;
        this.versionStamp = versionStamp;
        this.keystoreContent = keystoreContent;
        this.keyStoreTypeProvidedByStorage = keyStoreTypeProvidedByStorage;
        this.storageLevel = storageLevel;
        this.passwordPrefixBytes = new byte[64];
        RANDOM.nextBytes(this.passwordPrefixBytes);
        this.keyStore = this.getKeyStoreByStorageProvidedType(keyStoreTypeProvidedByStorage, keyStoreName);
    }

    private KeyStore getKeyStoreByStorageProvidedType(String keyStoreTypeProvidedByStorage, String keyStoreName) throws KeyStoreServiceException {
        KeyStore result;
        String keyStoreType = TypesMapper.getKeyStoreType(keyStoreTypeProvidedByStorage);
        if (LOG.isDebugEnabled()) {
            LOG.debug("KeyStore will be loaded with type '{}' mapped from storage provided type '{}'", (Object)keyStoreType, (Object)keyStoreTypeProvidedByStorage);
        }
        try {
            result = KeyStoreProvider.isKeyStoreTypeSupported(keyStoreType) ? KeyStore.getInstance(keyStoreType, "KeystoreService") : KeyStore.getInstance(keyStoreType);
        }
        catch (KeyStoreException e) {
            String resolveDetails = "";
            if (keyStoreType.equals(keyStoreTypeProvidedByStorage)) {
                resolveDetails = "' (resolved from '" + keyStoreTypeProvidedByStorage + "')";
            }
            throw new KeyStoreServiceException("Could not load keystore with name '" + keyStoreName + "', because the configured type: '" + keyStoreType + "'" + resolveDetails + "' is not supported", (Throwable)e);
        }
        catch (NoSuchProviderException e) {
            String resolveDetails = "";
            if (keyStoreType.equals(keyStoreTypeProvidedByStorage)) {
                resolveDetails = "' (resolved from '" + keyStoreTypeProvidedByStorage + "')";
            }
            throw new KeyStoreServiceException("Could not load keystore with name '" + keyStoreName + "' with type: '" + keyStoreType + "' " + resolveDetails + ", because the KeyStoreProvider is not registered", (Throwable)e);
        }
        return result;
    }

    public String getName() {
        return this.name;
    }

    public String getKeyStoreTypeProvidedByStorage() {
        return this.keyStoreTypeProvidedByStorage;
    }

    public KeyStore getKeyStore() {
        return this.keyStore;
    }

    public long getVersionStamp() {
        return this.versionStamp;
    }

    public StorageLevel getStorageLevel() {
        return this.storageLevel;
    }

    public String toString() {
        return "KeyStoreData [name=" + this.name + ", keyStore=" + this.keyStore + ", versionStamp=" + this.versionStamp + ", storage level=" + this.storageLevel.toString() + "]";
    }

    public void load(char[] password) throws KeyStoreServiceException {
        try {
            if (!this.isKeyStoreLoaded) {
                this.keyStore.load(new ByteArrayInputStream(this.keystoreContent), password);
                this.isKeyStoreLoaded = true;
                if (password != null) {
                    this.correctPasswordDigest = this.getDigest(password);
                }
            } else if (password != null) {
                byte[] digestOfPassword = this.getDigest(password);
                if (Arrays.equals(digestOfPassword, this.correctPasswordDigest)) {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Integrity check for keystore " + this.name + " was already done with the given password");
                    }
                } else {
                    KeyStore tempKeystore = KeyStore.getInstance(this.keyStore.getType());
                    tempKeystore.load(new ByteArrayInputStream(this.keystoreContent), password);
                    if (this.correctPasswordDigest == null) {
                        this.keyStore = tempKeystore;
                    }
                    this.correctPasswordDigest = digestOfPassword;
                }
            }
        }
        catch (NoSuchAlgorithmException e) {
            throw new KeyStoreServiceException("Could not load keystore with name: " + this.name + ", because the needed algorithm to check the integrity is not found", (Throwable)e);
        }
        catch (CertificateException e) {
            throw new KeyStoreServiceException("Could not load keystore with name: " + this.name + ", because some of the certificates in the keystore cannot be loaded", (Throwable)e);
        }
        catch (KeyStoreException e) {
            throw new KeyStoreServiceException("Could not load keystore with name: " + this.name + ", because of missing KeyStore implementation for type: " + this.keyStore.getType(), (Throwable)e);
        }
        catch (IOException e) {
            throw new KeyStoreServiceException("Could not load keystore with name: " + this.name + ", because of wrong format or wrong/missing password", (Throwable)e);
        }
    }

    private byte[] getDigest(char[] plaintext) throws KeyStoreServiceException {
        try {
            byte[] passwordBytes = new String(plaintext).getBytes("UTF-8");
            byte[] prefixedPassword = new byte[64 + passwordBytes.length];
            System.arraycopy(this.passwordPrefixBytes, 0, prefixedPassword, 0, this.passwordPrefixBytes.length);
            System.arraycopy(passwordBytes, 0, prefixedPassword, this.passwordPrefixBytes.length, passwordBytes.length);
            MessageDigest sha1 = MessageDigest.getInstance("SHA1");
            return sha1.digest(prefixedPassword);
        }
        catch (Exception e) {
            throw new KeyStoreServiceException("Could not digest the password", (Throwable)e);
        }
    }
}

