/*
 * Decompiled with CFR 0.152.
 */
package com.sap.cloud.crypto.keystore.service;

import com.sap.cloud.crypto.keystore.api.KeyStoreNotFoundException;
import com.sap.cloud.crypto.keystore.api.KeyStoreServiceException;
import com.sap.cloud.crypto.keystore.service.CacheAccessor;
import com.sap.cloud.crypto.keystore.service.ClientSideKeyStore;
import com.sap.cloud.crypto.keystore.service.ConfigurationUtil;
import com.sap.cloud.crypto.keystore.service.KeyStoreCloudStorage;
import com.sap.cloud.crypto.keystore.service.KeyStoreDomainDBStorage;
import com.sap.cloud.crypto.keystore.service.KeyStoreFileStorage;
import com.sap.cloud.crypto.keystore.service.KeyStoreStorage;
import com.sap.cloud.crypto.keystore.service.TenantProvider;
import com.sap.cloud.crypto.keystore.service.ext.KeyStoreAlreadyExistsException;
import com.sap.cloud.crypto.keystore.service.ext.KeyStoreServiceExt;
import com.sap.cloud.crypto.keystore.service.types.TypesMapper;
import java.io.ByteArrayOutputStream;
import java.security.KeyStore;
import java.util.HashSet;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KeyStoreServiceImpl
implements KeyStoreServiceExt {
    private static final int ONE_MEGABYTE = 0x100000;
    private static final String DOMAIN_DB_IS_NOT_AVAILABLE = "Domain DB is not available";
    private static final Logger LOG = LoggerFactory.getLogger(KeyStoreServiceImpl.class);
    private final TenantProvider tenantProvider;

    public KeyStoreServiceImpl() {
        this.tenantProvider = TenantProvider.forDynamicTenantForThread();
        LOG.debug("Initializing keystore service for dynamic tenant for thread");
    }

    public KeyStoreServiceImpl(String tenantName) {
        this.tenantProvider = TenantProvider.forPredefinedTenantName(tenantName);
        LOG.debug("Initializing keystore service for predefined tenant {}", (Object)tenantName);
    }

    public KeyStore getKeyStore(String keyStoreName, char[] password) throws KeyStoreServiceException, KeyStoreNotFoundException {
        if (keyStoreName == null || keyStoreName.length() == 0) {
            throw new KeyStoreServiceException("Keystore name is required");
        }
        ClientSideKeyStore result = null;
        try {
            result = CacheAccessor.getInstance().getCachedKeyStore(this.tenantProvider.getTenantName(), keyStoreName);
            if (result == null) {
                throw new KeyStoreNotFoundException("Keystore with name: '" + keyStoreName + "' is not found neither in the cloud (domain db) nor in the local (file) storage");
            }
            result.load(password);
        }
        catch (KeyStoreNotFoundException e) {
            LOG.warn(e.getMessage(), (Throwable)e);
            throw e;
        }
        catch (KeyStoreServiceException e) {
            LOG.error(e.getMessage(), (Throwable)e);
            throw e;
        }
        catch (Exception e) {
            String message = "Error occured while trying to retrieve keystore with name: '" + keyStoreName + "': " + e.getMessage();
            LOG.error(message, (Throwable)e);
            throw new KeyStoreServiceException(message, (Throwable)e);
        }
        return result;
    }

    public Set<String> getKeyStoreNames() {
        HashSet<String> result = new HashSet<String>();
        Set<String> localKeystoreNames = this.getKeyStoreLocalStorage().getKeyStoreNames();
        result.addAll(localKeystoreNames);
        Set<String> cloudKeystoreNames = null;
        if (this.isCloudStorageAvailable()) {
            cloudKeystoreNames = this.getKeyStoreCloudStorage().getKeyStoreNames();
            result.addAll(cloudKeystoreNames);
        }
        if (LOG.isDebugEnabled()) {
            StringBuilder message = new StringBuilder();
            message.append("Keystore names from local (file) storage: ").append(localKeystoreNames).append("\n");
            message.append("Keystore names from cloud storage: ");
            if (cloudKeystoreNames != null) {
                message.append(cloudKeystoreNames);
            }
            message.append("\n").append("Keystore names from all storages: ").append(result);
            LOG.debug(message.toString());
        }
        return result;
    }

    public void invalidateCache() {
        CacheAccessor.getInstance().invalidate(this.tenantProvider.getTenantName());
    }

    private KeyStoreStorage getKeyStoreLocalStorage() {
        return new KeyStoreFileStorage();
    }

    private KeyStoreCloudStorage getKeyStoreCloudStorage() {
        return new KeyStoreDomainDBStorage(this.tenantProvider.getTenantName());
    }

    private boolean isCloudStorageAvailable() {
        return ConfigurationUtil.isDomainDBAvailable();
    }

    @Override
    public void addKeyStoreMapping(String keystoreFileExtension, String keyStoreType) throws KeyStoreServiceException {
        if (keystoreFileExtension == null || (keystoreFileExtension = keystoreFileExtension.trim()).isEmpty()) {
            throw new IllegalArgumentException("Illegal argument - keystoreFileExtension must not be null or empty.");
        }
        if (keyStoreType == null || (keyStoreType = keyStoreType.trim()).isEmpty()) {
            throw new IllegalArgumentException("Illegal argument - key store type must not be null.");
        }
        if (keyStoreType.indexOf(" ") != -1) {
            throw new IllegalArgumentException("Illegal argument - key store type cannot contain spaces.");
        }
        TypesMapper.registerMapping(keystoreFileExtension, keyStoreType);
    }

    @Override
    public boolean removeKeyStoreMapping(String keystoreFileExtension, String keyStoreType) {
        if (keystoreFileExtension == null || (keystoreFileExtension = keystoreFileExtension.trim()).isEmpty()) {
            throw new IllegalArgumentException("Illegal argument - keystoreFileExtension must not be null or empty.");
        }
        return TypesMapper.unregisterMapping(keystoreFileExtension, keyStoreType);
    }

    @Override
    public void uploadKeyStore(String name, KeyStore keyStore, char[] password, boolean overwrite) throws KeyStoreAlreadyExistsException, KeyStoreServiceException {
        this.uploadKeyStoreAtSubscriptionLevel(name, keyStore, password, overwrite);
    }

    @Override
    public void uploadKeyStoreAtSubscriptionLevel(String name, KeyStore keyStore, char[] password, boolean overwrite) throws KeyStoreAlreadyExistsException, KeyStoreServiceException {
        byte[] keyStoreContent = this.getKeyStoreContent(name, keyStore, password);
        String fileExtension = this.getFileExtension(keyStore);
        this.getKeyStoreCloudStorage().uploadKeyStoreAtSubscriptionLevel(name, fileExtension, keyStoreContent, overwrite);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Upload to subscription: name: " + name + " overwrite:" + overwrite);
        }
    }

    @Override
    public void uploadKeyStoreAtApplicationLevel(String name, KeyStore keyStore, char[] password, boolean overwrite) throws KeyStoreAlreadyExistsException, KeyStoreServiceException {
        byte[] keyStoreContent = this.getKeyStoreContent(name, keyStore, password);
        String fileExtension = this.getFileExtension(keyStore);
        this.getKeyStoreCloudStorage().uploadKeyStoreAtApplicationLevel(name, fileExtension, keyStoreContent, overwrite);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Upload to application: name: " + name + " overwrite:" + overwrite);
        }
    }

    private byte[] getKeyStoreContent(String name, KeyStore keyStore, char[] password) throws IllegalArgumentException, KeyStoreServiceException {
        if (this.isEmpty(name)) {
            throw new IllegalArgumentException("Parameter " + name + " must be supplied");
        }
        if (keyStore == null) {
            throw new IllegalArgumentException("KeyStore must not be null");
        }
        if (!ConfigurationUtil.isDomainDBAvailable()) {
            throw new KeyStoreServiceException(DOMAIN_DB_IS_NOT_AVAILABLE);
        }
        byte[] keyStoreContent = this.getKeyStoreContent(keyStore, password);
        if (keyStoreContent.length > 0x100000) {
            throw new KeyStoreServiceException("The size of the keystore(bytes): " + keyStoreContent.length + " is bigger than maximum allowed: " + 0x100000);
        }
        if (keyStoreContent.length <= 0) {
            throw new KeyStoreServiceException("The given keystore instance is empty or its store(..,..) method is not implemented.");
        }
        return keyStoreContent;
    }

    private String getFileExtension(KeyStore keyStore) {
        String fileExtension = TypesMapper.getKeyStoreFileExtension(keyStore.getType());
        if (LOG.isDebugEnabled()) {
            LOG.debug("Keystore type returned by the keystore object '" + keyStore.getType() + "' was mapped to file extension '" + fileExtension + "'");
        }
        return fileExtension;
    }

    @Override
    public void deleteKeyStore(String name) throws KeyStoreNotFoundException, KeyStoreServiceException {
        this.deleteKeyStoreAtSubscriptionLevel(name);
    }

    @Override
    public void deleteKeyStoreAtSubscriptionLevel(String name) throws KeyStoreNotFoundException, KeyStoreServiceException {
        this.checkParamAndDomainDB(name);
        this.getKeyStoreCloudStorage().deleteKeyStoreAtSubscriptionLevel(name);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Delete at subscription level keystore with name: " + name);
        }
    }

    @Override
    public void deleteKeyStoreAtApplicationLevel(String name) throws KeyStoreNotFoundException, KeyStoreServiceException {
        this.checkParamAndDomainDB(name);
        this.getKeyStoreCloudStorage().deleteKeyStoreAtApplicationLevel(name);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Delete at application level keystore with name: " + name);
        }
    }

    private void checkParamAndDomainDB(String name) throws KeyStoreServiceException {
        if (this.isEmpty(name)) {
            throw new IllegalArgumentException("Parameter " + name + " must be supplied");
        }
        if (!ConfigurationUtil.isDomainDBAvailable()) {
            throw new KeyStoreServiceException(DOMAIN_DB_IS_NOT_AVAILABLE);
        }
    }

    private boolean isEmpty(String value) {
        return value == null || value.trim().length() == 0;
    }

    private byte[] getKeyStoreContent(KeyStore keyStore, char[] password) throws KeyStoreServiceException {
        try {
            ByteArrayOutputStream os = new ByteArrayOutputStream();
            keyStore.store(os, password);
            return os.toByteArray();
        }
        catch (Exception e) {
            throw new KeyStoreServiceException("Error occured while trying to get the content of the keystore", (Throwable)e);
        }
    }
}

