/*
 * Copyright (c) 2020 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.cache;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.Nonnull;
import javax.servlet.ServletContextEvent;
import javax.servlet.annotation.WebListener;

import com.google.common.cache.Cache;
import com.sap.cloud.sdk.cloudplatform.monitoring.JmxMonitor;

import lombok.Getter;
import lombok.NoArgsConstructor;

/**
 * Gives access to aggregated performance statistics for all caches statically managed by the CacheManager.
 */
@WebListener
@NoArgsConstructor
public class CacheMonitor extends JmxMonitor implements CacheMonitorMXBean
{
    @Getter
    private static final CacheMonitor instance = new CacheMonitor();

    @Override
    public long invalidateCaches()
    {
        return CacheManager.invalidateAll();
    }

    @Override
    public long invalidateTenantCaches( final String tenantId )
    {
        return CacheManager.invalidateTenantCaches(tenantId);
    }

    @Override
    public long invalidateUserCaches( final String tenantId, final String userName )
    {
        return CacheManager.invalidateUserCaches(tenantId, userName);
    }

    private CacheStats stats()
    {
        long cacheSize = 0;
        long hitCount = 0;
        long missCount = 0;
        long loadSuccessCount = 0;
        long loadExceptionCount = 0;
        long totalLoadTime = 0;
        long evictionCount = 0;

        for( final Cache<CacheKey, ?> cache : CacheManager.getCacheList() ) {
            final com.google.common.cache.CacheStats stats = cache.stats();

            cacheSize += cache.size();
            hitCount += stats.hitCount();
            missCount += stats.missCount();
            loadSuccessCount += stats.loadSuccessCount();
            loadExceptionCount += stats.loadExceptionCount();
            totalLoadTime += stats.totalLoadTime();
            evictionCount += stats.evictionCount();
        }

        return new CacheStats(
            cacheSize,
            hitCount,
            missCount,
            loadSuccessCount,
            loadExceptionCount,
            totalLoadTime,
            evictionCount);
    }

    @Override
    public long getCacheSize()
    {
        return stats().getCacheSize();
    }

    @Override
    public long getHitCount()
    {
        return stats().getHitCount();
    }

    @Override
    public long getMissCount()
    {
        return stats().getMissCount();
    }

    @Override
    public long getLoadSuccessCount()
    {
        return stats().getLoadSuccessCount();
    }

    @Override
    public long getLoadExceptionCount()
    {
        return stats().getLoadExceptionCount();
    }

    @Override
    public long getTotalLoadTime()
    {
        return stats().getTotalLoadTime();
    }

    @Override
    public long getEvictionCount()
    {
        return stats().getEvictionCount();
    }

    @Nonnull
    @Override
    public List<CacheStats> getCacheStatsList()
    {
        final List<CacheStats> result = new ArrayList<>();

        for( final Cache<CacheKey, ?> cache : CacheManager.getCacheList() ) {
            final com.google.common.cache.CacheStats stats = cache.stats();

            result.add(
                new CacheStats(
                    cache.size(),
                    stats.hitCount(),
                    stats.missCount(),
                    stats.loadSuccessCount(),
                    stats.loadExceptionCount(),
                    stats.totalLoadTime(),
                    stats.evictionCount()));
        }

        return result;
    }

    @Override
    public void contextInitialized( final ServletContextEvent servletContextEvent )
    {
        instance.registerJmxBean();
    }

    @Override
    public void contextDestroyed( final ServletContextEvent servletContextEvent )
    {
        instance.unregisterJmxBean();
    }
}
