/*
 * Copyright (c) 2020 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.cache;

import java.util.List;

import javax.annotation.Nonnull;

/**
 * JMX interface monitoring and invalidating caches as needed.
 */
public interface CacheMonitorMXBean
{
    /**
     * Invalidates all entries in all caches.
     *
     * @return The number of invalidated cache entries.
     */
    long invalidateCaches();

    /**
     * Invalidates all caches of the given tenant.
     *
     * @param tenantId
     *            The tenant to invalidate all caches for.
     *
     * @return The number of invalidated cache entries.
     */
    long invalidateTenantCaches( String tenantId );

    /**
     * Invalidates all caches of the given user.
     *
     * @param tenantId
     *            The Id of the tenant for which all user caches should be invalidated.
     * @param userName
     *            The name of the user for which all caches should be invalidated.
     *
     * @return The number of invalidated cache entries.
     */
    long invalidateUserCaches( String tenantId, String userName );

    /**
     * Returns the number of entries in all monitored caches.
     *
     * @return The number of all entries.
     */
    long getCacheSize();

    /**
     * Returns the number times cache lookup methods have returned a cached value.
     *
     * @return The number of successful cache lookups.
     */
    long getHitCount();

    /**
     * Returns the number of times cache lookup methods have returned an uncached value, or null.
     *
     * @return The number of cache lookup misses.
     */
    long getMissCount();

    /**
     * Returns the number of times cache lookup methods have successfully loaded a new value.
     *
     * @return The number of successful loads of new cache entries.
     */
    long getLoadSuccessCount();

    /**
     * Returns the number of times cache lookup methods threw an exception while loading a new value.
     *
     * @return The number of exceptions thrown while loading new values.
     */
    long getLoadExceptionCount();

    /**
     * Returns the total number of nanoseconds the cache has spent loading new values.
     *
     * @return The total time spent loading new values, in nanoseconds.
     */
    long getTotalLoadTime();

    /**
     * Returns the number of times an entry has been automatically evicted.
     *
     * @return The number of automatic eviction of entries.
     */
    long getEvictionCount();

    /**
     * Returns a list containing a {@link CacheStats} instance for each monitored cache.
     * 
     * @return A list of one stats object pre cache.
     */
    @Nonnull
    List<CacheStats> getCacheStatsList();
}
