/*
 * Copyright (c) 2020 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.concurrency;

import java.util.concurrent.Callable;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.servlet.http.HttpServletRequest;

import org.slf4j.Logger;

import com.sap.cloud.sdk.cloudplatform.logging.CloudLoggerFactory;
import com.sap.cloud.sdk.cloudplatform.security.user.ScpNeoUser;
import com.sap.cloud.sdk.cloudplatform.servlet.RequestContext;
import com.sap.marmolata.hcp.backend.HCPInterop$;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.ToString;

/**
 * Wraps the given {@link Callable} to be executed in the context of the current user (if available).
 */
@RequiredArgsConstructor
@Getter
@ToString
public class ScpNeoUserSessionCallable<T> implements Callable<T>
{
    private static final Logger logger = CloudLoggerFactory.getLogger(ScpNeoUserSessionCallable.class);

    @Nullable
    private final HttpServletRequest request;

    @Nonnull
    private final ScpNeoUser currentUser;

    @Nonnull
    private final Callable<T> callable;

    @SuppressWarnings( "PMD.SignatureDeclareThrowsException" )
    @Nullable
    @Override
    public T call()
        throws Exception
    {
        if( request == null ) {
            if( logger.isDebugEnabled() ) {
                logger.debug(
                    "Executing user session callable in context of thread "
                        + Thread.currentThread().getId()
                        + " without user context (no "
                        + RequestContext.class.getSimpleName()
                        + " available).");
            }

            return callable.call();
        }

        if( logger.isDebugEnabled() ) {
            logger.debug(
                "Executing user session callable in context of thread "
                    + Thread.currentThread().getId()
                    + " and user "
                    + currentUser
                    + ".");
        }

        final HCPInterop$ hcpInterop = new HCPInterop$();

        try {
            hcpInterop.initializeUserSession(request);
            try {
                hcpInterop.initializeUserSessionAccessor(request);
                try {
                    return callable.call();
                }
                // CHECKSTYLE:OFF
                catch( final Throwable t ) {
                    logger.error(
                        "An error occurred while running callable of user "
                            + currentUser
                            + ": "
                            + t.getMessage()
                            + ".");
                    throw t;
                }
                // CHECKSTYLE:ON
                finally {
                    hcpInterop.resetUserSessionAccessor();
                }
            }
            // CHECKSTYLE:OFF
            catch( final Throwable t ) {
                logger.error(
                    "An error occurred while running callable of user " + currentUser + ": " + t.getMessage() + ".");
                throw t;
            }
            // CHECKSTYLE:ON
            finally {
                hcpInterop.resetUserSession();
            }
        }
        catch( final ClassNotFoundException | NoSuchMethodException e ) {
            throw new AssertionError(
                "Failed to correctly wrap "
                    + ScpNeoUserSessionCallable.class.getSimpleName()
                    + ". "
                    + "This indicates a misconfiguration. "
                    + "Make sure to add 'com.sap.security.auth.service' and 'com.sap.security.um.service.api' "
                    + "to the Import-Package manifest entry of the WAR bundle. "
                    + "In Maven, you can achieve this by adding the following configuration to the maven-war-plugin: "
                    + "<plugin>"
                    + "<groupId>org.apache.maven.plugins</groupId>"
                    + "<artifactId>maven-war-plugin</artifactId>"
                    + "<configuration>"
                    + "<archive>"
                    + "<manifestEntries>"
                    + "<Import-Package>com.sap.security.auth.service,com.sap.security.um.service.api</Import-Package>"
                    + "</manifestEntries>"
                    + "</archive>"
                    + "</configuration>"
                    + "</plugin>.",
                e);
        }
    }
}
