/*
 * Copyright (c) 2020 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security.secret;

import javax.annotation.Nonnull;

import com.sap.cloud.sdk.cloudplatform.security.secret.exception.SecretStoreAccessException;
import com.sap.cloud.security.password.PasswordStorage;
import com.sap.cloud.security.password.PasswordStorageException;

import lombok.RequiredArgsConstructor;

/**
 * Implementation of {@link SecretStore} for SAP Cloud Platform Neo.
 */
@RequiredArgsConstructor
public class ScpNeoSecretStore implements SecretStore
{
    @Nonnull
    private final PasswordStorage passwordStorage;

    @Nonnull
    private final String name;

    /**
     * Mocked {@link PasswordStorage} that does not return any password.
     */
    private static class MockedPasswordStorage implements PasswordStorage
    {
        @Override
        public char[] getPassword( final String s )
            throws PasswordStorageException
        {
            return null;
        }

        @Override
        public void setPassword( final String s, final char[] chars )
            throws PasswordStorageException
        {

        }

        @Override
        public void deletePassword( final String s )
            throws PasswordStorageException
        {

        }
    }

    /**
     * Creates a mocked {@link ScpNeoSecretStore} with an empty name.
     * <p>
     * This no-arguments constructor is required to ensure compatibility with mocking frameworks such as Mockito.
     */
    private ScpNeoSecretStore()
    {
        passwordStorage = new MockedPasswordStorage();
        name = "";
    }

    @Override
    public char[] getSecret()
        throws SecretStoreAccessException
    {
        try {
            return passwordStorage.getPassword(name);
        }
        catch( final PasswordStorageException | IllegalArgumentException e ) {
            throw new SecretStoreAccessException("Failed to access secret store.", e);
        }
    }
}
