/*
 * Copyright (c) 2020 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security.secret;

import java.security.KeyStore;
import java.util.Optional;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;

import com.sap.cloud.crypto.keystore.api.KeyStoreService;
import com.sap.cloud.crypto.keystore.api.KeyStoreServiceException;
import com.sap.cloud.sdk.cloudplatform.logging.CloudLoggerFactory;
import com.sap.cloud.sdk.cloudplatform.security.secret.exception.KeyStoreAccessException;
import com.sap.cloud.sdk.cloudplatform.security.secret.exception.SecretStoreAccessException;
import com.sap.cloud.sdk.cloudplatform.servlet.Property;
import com.sap.cloud.sdk.cloudplatform.servlet.RequestContext;
import com.sap.cloud.sdk.cloudplatform.servlet.RequestContextAccessor;
import com.sap.cloud.sdk.cloudplatform.servlet.RequestContextExecutor;
import com.sap.cloud.sdk.cloudplatform.servlet.RequestContextServletFilter;
import com.sap.cloud.sdk.cloudplatform.servlet.exception.RequestContextPropertyException;
import com.sap.cloud.security.password.PasswordStorage;
import com.sap.cloud.security.password.PasswordStorageException;

/**
 * Implementation of {@link SecretStoreFacade} for SAP Cloud Platform Neo.
 */
public class ScpNeoSecretStoreFacade extends AbstractSecretStoreFacade
{
    private static final Logger logger = CloudLoggerFactory.getLogger(ScpNeoSecretStoreFacade.class);

    private PasswordStorage getPasswordStorage()
        throws SecretStoreAccessException
    {
        final Optional<RequestContext> requestContext = RequestContextAccessor.getCurrentRequestContext();

        if( !requestContext.isPresent() ) {
            throw new SecretStoreAccessException(
                "Failed to get "
                    + PasswordStorage.class.getSimpleName()
                    + ": no "
                    + RequestContext.class.getSimpleName()
                    + " available."
                    + " Have you correctly configured a "
                    + RequestContextServletFilter.class.getSimpleName()
                    + " or have you wrapped your logic in a "
                    + RequestContextExecutor.class.getSimpleName()
                    + " when executing background tasks that are not triggered by a request?");
        }

        try {
            final Optional<Property<?>> property =
                requestContext.get().getProperty(ScpNeoSecretStoreRequestContextListener.PROPERTY_PASSWORD_STORAGE);

            if( !property.isPresent() ) {
                throw new SecretStoreAccessException(
                    "Failed to get "
                        + PasswordStorage.class.getSimpleName()
                        + ": "
                        + RequestContext.class.getSimpleName()
                        + " property '"
                        + ScpNeoSecretStoreRequestContextListener.PROPERTY_PASSWORD_STORAGE
                        + "' is not present. "
                        + "Please ensure that "
                        + ScpNeoSecretStoreRequestContextListener.class.getSimpleName()
                        + " is available on the class path.");
            }

            @Nullable
            final Exception exception = property.get().getException();
            if( exception != null ) {
                throw new SecretStoreAccessException(
                    "Failed to get " + PasswordStorage.class.getSimpleName() + ".",
                    exception);
            }

            return (PasswordStorage) property.get().getValue();
        }
        catch( final RequestContextPropertyException e ) {
            throw new SecretStoreAccessException(
                "Failed to get "
                    + PasswordStorage.class.getSimpleName()
                    + ": failed to get "
                    + RequestContext.class.getSimpleName()
                    + " property.",
                e);
        }
    }

    private KeyStoreService getKeyStoreService()
        throws KeyStoreAccessException
    {
        final Optional<RequestContext> requestContext = RequestContextAccessor.getCurrentRequestContext();

        if( !requestContext.isPresent() ) {
            throw new KeyStoreAccessException(
                "Failed to get "
                    + KeyStoreService.class.getSimpleName()
                    + ": no "
                    + RequestContext.class.getSimpleName()
                    + " available."
                    + " Have you correctly configured a "
                    + RequestContextServletFilter.class.getSimpleName()
                    + " or have you wrapped your logic in a "
                    + RequestContextExecutor.class.getSimpleName()
                    + " when executing background tasks that are not triggered by a request?");
        }

        try {
            final Optional<Property<?>> property =
                requestContext.get().getProperty(ScpNeoSecretStoreRequestContextListener.PROPERTY_KEY_STORE_SERVICE);

            if( !property.isPresent() ) {
                throw new KeyStoreAccessException(
                    "Failed to get "
                        + KeyStoreService.class.getSimpleName()
                        + ": "
                        + RequestContext.class.getSimpleName()
                        + " property '"
                        + ScpNeoSecretStoreRequestContextListener.PROPERTY_KEY_STORE_SERVICE
                        + "' is not present. "
                        + "Please ensure that "
                        + ScpNeoSecretStoreRequestContextListener.class.getSimpleName()
                        + " is available on the class path.");
            }

            @Nullable
            final Exception exception = property.get().getException();
            if( exception != null ) {
                throw new KeyStoreAccessException(
                    "Failed to get " + KeyStoreService.class.getSimpleName() + ".",
                    exception);
            }

            return (KeyStoreService) property.get().getValue();
        }
        catch( final RequestContextPropertyException e ) {
            throw new KeyStoreAccessException(
                "Failed to get "
                    + KeyStoreService.class.getSimpleName()
                    + ": failed to get "
                    + RequestContext.class.getSimpleName()
                    + " property.",
                e);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Nonnull
    @Override
    public Class<? extends SecretStore> getSecretStoreClass()
    {
        return ScpNeoSecretStore.class;
    }

    /**
     * {@inheritDoc}
     */
    @Nonnull
    @Override
    public Optional<SecretStore> getSecretStoreIfPresent( final String name )
        throws SecretStoreAccessException
    {
        final PasswordStorage passwordStorage = getPasswordStorage();

        try {
            if( passwordStorage.getPassword(name) == null ) {
                if( logger.isDebugEnabled() ) {
                    logger.debug("No secret store found with name '" + name + "'.");
                }
                return Optional.empty();
            }
        }
        catch( final PasswordStorageException | IllegalArgumentException e ) {
            throw new SecretStoreAccessException("Failed to access secret store.", e);
        }

        return Optional.of(new ScpNeoSecretStore(passwordStorage, name));
    }

    /**
     * {@inheritDoc}
     */
    @Nonnull
    @Override
    public Optional<KeyStore> getKeyStoreIfPresent( final String name, final SecretStore password )
        throws KeyStoreAccessException
    {
        final KeyStoreService keyStoreService = getKeyStoreService();

        if( !keyStoreService.getKeyStoreNames().contains(name) ) {
            if( logger.isDebugEnabled() ) {
                logger.debug("No key store found with name '" + name + "'.");
            }
            return Optional.empty();
        }

        try {
            return Optional.of(keyStoreService.getKeyStore(name, password.getSecret()));
        }
        catch( final KeyStoreServiceException e ) {
            throw new KeyStoreAccessException("Failed to access key store.", e);
        }
    }
}
