/*
 * Copyright (c) 2020 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security.secret;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;

import com.sap.cloud.sdk.cloudplatform.exception.ObjectLookupFailedException;
import com.sap.cloud.sdk.cloudplatform.naming.JndiLookupAccessor;
import com.sap.cloud.sdk.cloudplatform.servlet.AbstractRequestContextListener;
import com.sap.cloud.sdk.cloudplatform.servlet.Property;
import com.sap.cloud.sdk.cloudplatform.servlet.RequestContext;

/**
 * Implementation of {@link AbstractRequestContextListener} that performs the JNDI lookup of the password storage and
 * key store service objects on SAP CP Neo.
 */
public class ScpNeoSecretStoreRequestContextListener extends AbstractRequestContextListener
{
    public static final String PROPERTY_PASSWORD_STORAGE =
        ScpNeoSecretStoreRequestContextListener.class.getName() + ":passwordStorage";
    public static final String PROPERTY_KEY_STORE_SERVICE =
        ScpNeoSecretStoreRequestContextListener.class.getName() + ":keyStoreService";

    private static final String JNDI_NAME_PASSWORD_STORAGE = "java:comp/env/PasswordStorage";
    private static final String JNDI_NAME_KEY_STORE_SERVICE = "java:comp/env/KeyStoreService";

    @Override
    public int getPriority()
    {
        return DefaultPriorities.CUSTOM_LISTENER + 3;
    }

    @Nonnull
    @Override
    protected Map<String, Property<?>> getProperties( @Nonnull final RequestContext requestContext )
    {
        final Map<String, Property<?>> properties = new HashMap<>();

        Property<?> passwordStorage;
        Property<?> keyStoreService;

        try {
            passwordStorage = Property.ofConfidentialValue(JndiLookupAccessor.lookup(JNDI_NAME_PASSWORD_STORAGE));
            keyStoreService = Property.ofConfidentialValue(JndiLookupAccessor.lookup(JNDI_NAME_KEY_STORE_SERVICE));
        }
        catch( final ObjectLookupFailedException e ) {
            passwordStorage = Property.ofConfidentialException(e);
            keyStoreService = Property.ofConfidentialException(e);
        }

        properties.put(PROPERTY_PASSWORD_STORAGE, passwordStorage);
        properties.put(PROPERTY_KEY_STORE_SERVICE, keyStoreService);

        return properties;
    }
}
