/*
 * Copyright (c) 2020 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security.servlet;

import java.io.IOException;

import javax.annotation.Nullable;
import javax.security.auth.login.LoginContext;
import javax.security.auth.login.LoginException;
import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.slf4j.Logger;

import com.sap.cloud.sdk.cloudplatform.auditlog.AuditLogger;
import com.sap.cloud.sdk.cloudplatform.exception.ShouldNotHappenException;
import com.sap.cloud.sdk.cloudplatform.logging.CloudLoggerFactory;
import com.sap.security.auth.login.LoginContextFactory;

/**
 * Filter rejecting all non-HTTP requests and responses as well as log in the remote user.
 */
public class AuthContextFilter implements Filter
{
    private static final Logger logger = CloudLoggerFactory.getLogger(AuthContextFilter.class);

    private static final String INIT_PARAMETER_AUTH_METHOD = "method";
    private String authMethod = null;

    @Override
    public void init( final FilterConfig filterConfig )
    {
        authMethod = filterConfig.getInitParameter(INIT_PARAMETER_AUTH_METHOD);
    }

    @Override
    public void doFilter(
        final ServletRequest servletRequest,
        final ServletResponse servletResponse,
        final FilterChain filterChain )
        throws IOException,
            ServletException
    {
        if( !(servletResponse instanceof HttpServletResponse) ) {
            return;
        }

        if( !(servletRequest instanceof HttpServletRequest) ) {
            return;
        }

        final HttpServletRequest req = (HttpServletRequest) servletRequest;
        final HttpServletResponse resp = (HttpServletResponse) servletResponse;

        final String user = req.getRemoteUser();

        if( user != null ) {
            filterChain.doFilter(req, resp);
        } else {
            final LoginContext loginContext;
            try {
                if( authMethod == null ) {
                    throw new ShouldNotHappenException(
                        "Authorization filter misconfiguration: missing authentication method. "
                            + "Please specify the initialization parameter '"
                            + INIT_PARAMETER_AUTH_METHOD
                            + "'.");
                }

                // invalidate session to avoid possible session fixation vulnerability
                @Nullable
                final HttpSession session = req.getSession(false);
                if( session != null ) {
                    session.invalidate();
                }

                loginContext = LoginContextFactory.createLoginContext(authMethod);
                loginContext.login();

                filterChain.doFilter(req, resp);
            }
            catch( final LoginException e ) {
                resp.setStatus(HttpServletResponse.SC_UNAUTHORIZED);
                AuditLogger.logSecurityEvent("Unauthorized login attempt. Message: " + e.getMessage() + ".", null);

                if( logger.isWarnEnabled() ) {
                    logger.warn(e.getMessage(), e);
                }
            }
        }
    }

    @Override
    public void destroy()
    {

    }
}
