/*
 * Copyright (c) 2020 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security.servlet;

import java.io.IOException;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.http.HttpStatus;
import org.slf4j.Logger;

import com.sap.cloud.sdk.cloudplatform.CloudPlatform;
import com.sap.cloud.sdk.cloudplatform.CloudPlatformAccessor;
import com.sap.cloud.sdk.cloudplatform.ScpNeoCloudPlatform;
import com.sap.cloud.sdk.cloudplatform.auditlog.AuditLogger;
import com.sap.cloud.sdk.cloudplatform.exception.ShouldNotHappenException;
import com.sap.cloud.sdk.cloudplatform.logging.CloudLoggerFactory;
import com.sap.cloud.sdk.cloudplatform.tenant.ScpNeoTenant;
import com.sap.cloud.sdk.cloudplatform.tenant.TenantAccessor;

/**
 * Filter that only allows HTTP requests that are executed within the context of the provider account in the SAP Cloud
 * Platforn Neo environment.
 */
public class ProviderAccountFilter implements Filter
{
    private static final Logger logger = CloudLoggerFactory.getLogger(ProviderAccountFilter.class);

    @Override
    public void init( final FilterConfig config )
    {
    }

    @Override
    public void doFilter( final ServletRequest request, final ServletResponse response, final FilterChain chain )
        throws IOException,
            ServletException
    {
        if( !(response instanceof HttpServletResponse) ) {
            return;
        }

        if( !(request instanceof HttpServletRequest) ) {
            return;
        }

        final HttpServletResponse resp = (HttpServletResponse) response;

        final ScpNeoTenant tenant = ((ScpNeoTenant) TenantAccessor.getCurrentTenant());
        final String accountId = tenant.getTenantContext().getTenant().getAccount().getId();

        final CloudPlatform cloudPlatform = CloudPlatformAccessor.getCloudPlatform();

        if( !(cloudPlatform instanceof ScpNeoCloudPlatform) ) {
            throw new ShouldNotHappenException(
                "The current Cloud platform is not an instance of "
                    + ScpNeoCloudPlatform.class.getSimpleName()
                    + ". Please make sure to specify a dependency to com.sap.cloud.s4hana.cloudplatform:core-scp-neo.");
        }

        final ScpNeoCloudPlatform scpNeoCloudPlatform = (ScpNeoCloudPlatform) cloudPlatform;

        if( scpNeoCloudPlatform.getProviderAccountId().equals(accountId) ) {
            chain.doFilter(request, response);
        } else {
            resp.setStatus(HttpStatus.SC_UNAUTHORIZED);
            final String message =
                "Unauthorized attempt to access resource that must " + "only be accessed from within provider account.";

            AuditLogger.logSecurityEvent(message, null);
            logger.error(message);
        }
    }

    @Override
    public void destroy()
    {

    }
}
