/*
 * Copyright (c) 2020 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security.user;

import java.util.Calendar;
import java.util.Collections;
import java.util.Locale;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;

import com.sap.cloud.sdk.cloudplatform.logging.CloudLoggerFactory;
import com.sap.cloud.sdk.cloudplatform.security.Authorization;
import com.sap.cloud.sdk.cloudplatform.security.Role;
import com.sap.cloud.sdk.cloudplatform.security.user.exception.UserAccessException;
import com.sap.security.um.user.PasswordCheckResult;
import com.sap.security.um.user.UnsupportedUserAttributeException;

import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * Implementation of {@link User} for SAP Cloud Platform Neo.
 */
@EqualsAndHashCode( callSuper = true, exclude = "delegate" )
@ToString( callSuper = true )
@Data
public class ScpNeoUser extends AbstractUser
{
    private static final Logger logger = CloudLoggerFactory.getLogger(ScpNeoUser.class);

    public static final String FIRST_NAME = "firstname";
    public static final String LAST_NAME = "lastname";
    public static final String EMAIL = "email";

    private static final Function<String, Authorization> transformRole = new Function<String, Authorization>()
    {
        @Nullable
        @Override
        public Authorization apply( @Nullable final String roleName )
        {
            if( roleName == null ) {
                return null;
            }

            return new Role(roleName);
        }
    };

    /**
     * Mocked {@link com.sap.security.um.user.User} returning an empty user name with no authorizations.
     */
    private static class MockedUser implements com.sap.security.um.user.User
    {
        @SuppressWarnings( "deprecation" )
        @Override
        public PasswordCheckResult checkPassword( final char[] chars )
        {
            return PasswordCheckResult.FAILED_UNKNOWN_REASON;
        }

        @Override
        public String getAttribute( final String s )
        {
            return null;
        }

        @Override
        public Set<String> listAttributes()
        {
            return Collections.emptySet();
        }

        @Override
        public Locale getLocale()
        {
            return null;
        }

        @Override
        public boolean hasRole( final String s )
        {
            return false;
        }

        @Override
        public boolean isValid( final Calendar calendar )
        {
            return false;
        }

        @Override
        public Set<String> getRoles()
        {
            return Collections.emptySet();
        }

        @Override
        public String[] getAttributeValues( final String s )
        {
            return null;
        }

        @Override
        public Set<String> getGroups()
        {
            return Collections.emptySet();
        }

        @Override
        public String getName()
        {
            return "";
        }
    }

    @Nonnull
    static UserProviderAccessor userProviderAccessor = new UserProviderAccessor();

    /**
     * Delegate to the current {@link com.sap.security.um.user.User}.
     */
    @Nonnull
    private final com.sap.security.um.user.User delegate;

    /**
     * Creates a new {@link ScpNeoUser}.
     *
     * @param delegate
     *            The user to delegate to.
     */
    public ScpNeoUser( @Nonnull final com.sap.security.um.user.User delegate )
    {
        super(
            delegate.getName().toLowerCase(Locale.ENGLISH),
            delegate.getLocale(),
            Collections.emptySet(),
            Collections.emptyMap());

        this.delegate = delegate;
    }

    /**
     * Creates a new {@link ScpNeoUser} based on the current {@link com.sap.security.um.user.User}.
     *
     * @throws UserAccessException
     *             If there is an issue while accessing the current {@link com.sap.security.um.user.User}.
     */
    @Nullable
    static ScpNeoUser ofCurrentUser( @Nonnull final String name )
        throws UserAccessException
    {
        final com.sap.security.um.user.User delegate = userProviderAccessor.getUser(name);

        if( delegate == null ) {
            return null;
        }

        return new ScpNeoUser(delegate);
    }

    /**
     * Creates a mocked {@link ScpNeoUser} returning an empty user name and no authorizations.
     */
    static ScpNeoUser ofMockedUser()
    {
        return new ScpNeoUser();
    }

    /**
     * Creates a mocked {@link ScpNeoUser} returning an empty user name and no authorizations.
     */
    private ScpNeoUser()
    {
        this(new MockedUser());
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean hasAuthorization( @Nonnull final Authorization authorization )
    {
        return delegate.hasRole(authorization.getName());
    }

    /**
     * {@inheritDoc}
     */
    @Nonnull
    @Override
    public Set<Authorization> getAuthorizations()
    {
        return delegate.getRoles().stream().map(transformRole).collect(Collectors.toSet());
    }

    /**
     * {@inheritDoc}
     */
    @Nonnull
    @Override
    public Optional<UserAttribute> getAttribute( @Nonnull final String name )
    {
        try {
            final String attribute = delegate.getAttribute(name);

            if( attribute == null ) {
                return Optional.empty();
            }

            final UserAttribute userAttribute = new StringUserAttribute(name, attribute);
            return Optional.of(userAttribute);
        }
        catch( final UnsupportedUserAttributeException e ) {
            if( logger.isWarnEnabled() ) {
                logger.warn("Failed to get user attribute " + name + ".", e);
            }
            return Optional.empty();
        }
    }
}
