/*
 * Copyright (c) 2020 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security.user;

import java.util.Locale;
import java.util.Optional;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.servlet.http.HttpServletRequest;

import org.slf4j.Logger;

import com.sap.cloud.sdk.cloudplatform.WithRuntimeDependencies;
import com.sap.cloud.sdk.cloudplatform.exception.DependencyNotFoundException;
import com.sap.cloud.sdk.cloudplatform.logging.CloudLoggerFactory;
import com.sap.cloud.sdk.cloudplatform.security.user.exception.UserAccessDeniedException;
import com.sap.cloud.sdk.cloudplatform.security.user.exception.UserAccessException;
import com.sap.cloud.sdk.cloudplatform.servlet.RequestContext;
import com.sap.cloud.sdk.cloudplatform.servlet.RequestContextAccessor;
import com.sap.cloud.sdk.cloudplatform.servlet.RequestContextExecutor;
import com.sap.cloud.sdk.cloudplatform.servlet.RequestContextServletFilter;

/**
 * Facade providing access to user information on SAP Cloud Platform Neo.
 */
public class ScpNeoUserFacade extends AbstractUserFacade implements WithRuntimeDependencies
{
    private static final Logger logger = CloudLoggerFactory.getLogger(ScpNeoUserFacade.class);

    /**
     * {@inheritDoc}
     */
    @Nonnull
    @Override
    public Class<ScpNeoUser> getUserClass()
    {
        return ScpNeoUser.class;
    }

    /**
     * {@inheritDoc}
     */
    @Nonnull
    @Override
    protected User newMockedUser()
    {
        return ScpNeoUser.ofMockedUser();
    }

    @Nonnull
    private String getUserName( final HttpServletRequest request )
    {
        return request.getUserPrincipal().getName().toLowerCase(Locale.ENGLISH);
    }

    @Nullable
    private String getCurrentUserName()
        throws UserAccessException
    {
        final Optional<RequestContext> requestContext = RequestContextAccessor.getCurrentRequestContext();

        if( !requestContext.isPresent() ) {
            throw new UserAccessException(
                "Failed to get current user name: no "
                    + RequestContext.class.getSimpleName()
                    + " available."
                    + " Have you correctly configured a "
                    + RequestContextServletFilter.class.getSimpleName()
                    + " or have you wrapped your logic in a "
                    + RequestContextExecutor.class.getSimpleName()
                    + " when executing background tasks that are not triggered by a request?");
        }

        final Optional<HttpServletRequest> request = requestContext.get().getRequest();

        if( !request.isPresent() ) {
            if( logger.isDebugEnabled() ) {
                logger.debug(
                    "Unable to get current user: no request available. "
                        + "This is expected when trying to obtain the current user "
                        + "within tasks that are not triggered by a request, "
                        + "for example, while using a "
                        + RequestContextExecutor.class.getSimpleName()
                        + ".");
            }
            return null;
        }

        if( request.get().getUserPrincipal() == null ) {
            if( logger.isDebugEnabled() ) {
                logger.debug("Unable to get user name from request: user principal is null.");
            }
            return null;
        }

        return getUserName(request.get());
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void assertRuntimeDependenciesExist()
        throws DependencyNotFoundException
    {
        try {
            Class.forName("com.sap.security.um.user.User");
            Class.forName("com.sap.security.um.service.UserManagementAccessor");
        }
        catch( final ClassNotFoundException e ) {
            throw new DependencyNotFoundException(
                "Failed to instantiate User. "
                    + "Please make sure to use the local SAP Cloud Platform Neo runtime. "
                    + "If you require another container for local deployment, "
                    + "make sure to add the latest version of one of the following dependencies to your project: "
                    + "'com.sap.cloud:neo-java-web-api' for a Tomcat container or "
                    + "'com.sap.cloud:neo-javaee7-wp-api' for a Java EE container such as TomEE. "
                    + "Note that this requires separate Maven profiles: "
                    + "a profile for local development that uses the respective dependency with scope 'compile' and "
                    + "a profile for deployment on SAP Cloud Platform Neo that uses scope 'provided'.",
                e);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Nonnull
    @Override
    public Optional<User> resolveCurrentUser()
        throws UserAccessException
    {
        assertRuntimeDependenciesExist();

        final Optional<User> mockedUser = getMockedUser();

        if( mockedUser.isPresent() ) {
            return mockedUser;
        }

        @Nullable
        final String name = getCurrentUserName();

        if( name == null ) {
            return Optional.empty();
        }

        return Optional.ofNullable(ScpNeoUser.ofCurrentUser(name));
    }

    /**
     * {@inheritDoc}
     */
    @Nonnull
    @Override
    public Optional<User> getUserByName( @Nullable final String name )
        throws UserAccessDeniedException,
            UserAccessException
    {
        assertRuntimeDependenciesExist();

        if( name == null ) {
            return Optional.empty();
        }

        final Optional<User> mockedUser = getMockedUser();

        if( mockedUser.isPresent() && name.equals(mockedUser.get().getName()) ) {
            return mockedUser;
        }

        return Optional.ofNullable(ScpNeoUser.ofCurrentUser(name));
    }
}
