/*
 * Copyright (c) 2019 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security.principal;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.sap.cloud.sdk.cloudplatform.exception.ShouldNotHappenException;
import com.sap.cloud.sdk.cloudplatform.util.FacadeLocator;

/**
 * Main class find and access {@link Principal}.
 * <p>
 * This class handles the access to a {@code Principal} by delegating the calls to the underlying
 * {@link PrincipalFacade}.
 */
public class PrincipalAccessor
{
    @Nullable
    private static PrincipalFacade principalFacade = FacadeLocator.getFacade(PrincipalFacade.class);

    /**
     * Returns the {@link PrincipalFacade} instance.
     *
     * @throws ShouldNotHappenException
     *             If the {@link PrincipalFacade} instance is null.
     */
    @Nonnull
    static PrincipalFacade facade()
        throws ShouldNotHappenException
    {
        if( principalFacade == null ) {
            throw new ShouldNotHappenException(
                "Failed to determine the current Cloud platform while accessing principal information. "
                    + FacadeLocator.UNKNOWN_CLOUD_PLATFORM_MESSAGE);
        }

        return principalFacade;
    }

    /**
     * Replaces the default {@link PrincipalFacade} instance. This method is for internal use only.
     *
     * @param principalFacade
     *            An instance of {@link PrincipalFacade}.
     */
    public static void setPrincipalFacade( @Nonnull final PrincipalFacade principalFacade )
    {
        PrincipalAccessor.principalFacade = principalFacade;
    }

    /**
     * Gets the currently authenticated Principal.
     *
     * @return The current Principal.
     * @throws PrincipalAccessException
     *             if an error occurred while retrieving resources necessary to construct a {@code Principal}.
     */
    @Nonnull
    public static Principal getCurrentPrincipal()
        throws PrincipalAccessException
    {
        return facade().getCurrentPrincipal();
    }
}
