/*
 * Copyright (c) 2019 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security.secret;

import java.security.KeyStore;
import java.util.Optional;

import javax.annotation.Nonnull;

import com.sap.cloud.sdk.cloudplatform.security.secret.exception.KeyStoreAccessException;
import com.sap.cloud.sdk.cloudplatform.security.secret.exception.KeyStoreNotFoundException;
import com.sap.cloud.sdk.cloudplatform.security.secret.exception.SecretStoreAccessException;
import com.sap.cloud.sdk.cloudplatform.security.secret.exception.SecretStoreNotFoundException;

/**
 * Abstract base class for implementations of {@link SecretStoreFacade}.
 */
public abstract class AbstractSecretStoreFacade implements SecretStoreFacade
{
    /**
     * {@inheritDoc}
     */
    @Nonnull
    @Override
    public SecretStore getSecretStore( final String name )
        throws SecretStoreNotFoundException,
            SecretStoreAccessException
    {
        final Optional<SecretStore> secretStore = getSecretStoreIfPresent(name);

        if( !secretStore.isPresent() ) {
            throw new SecretStoreNotFoundException("Failed to find secret store with name '" + name + "'.");
        }

        return secretStore.get();
    }

    /**
     * {@inheritDoc}
     */
    @Nonnull
    @Override
    public KeyStore getKeyStore( final String name, final SecretStore password )
        throws KeyStoreNotFoundException,
            KeyStoreAccessException
    {
        final Optional<KeyStore> keyStore = getKeyStoreIfPresent(name, password);

        if( !keyStore.isPresent() ) {
            throw new KeyStoreNotFoundException("Failed to find key store with name '" + name + "'.");
        }

        return keyStore.get();
    }
}
