/*
 * Copyright (c) 2019 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security.secret;

import java.security.KeyStore;
import java.util.Optional;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.sap.cloud.sdk.cloudplatform.exception.ShouldNotHappenException;
import com.sap.cloud.sdk.cloudplatform.security.secret.exception.KeyStoreAccessException;
import com.sap.cloud.sdk.cloudplatform.security.secret.exception.KeyStoreNotFoundException;
import com.sap.cloud.sdk.cloudplatform.security.secret.exception.SecretStoreAccessException;
import com.sap.cloud.sdk.cloudplatform.security.secret.exception.SecretStoreNotFoundException;
import com.sap.cloud.sdk.cloudplatform.util.FacadeLocator;

import lombok.Getter;

/**
 * Main class for the secret store functionality.
 * <p>
 * This class handles the access to the secret and key store by delegating the calls to the underlying
 * {@link SecretStoreFacade}.
 */
public class SecretStoreAccessor
{
    /**
     * Returns the {@link SecretStoreFacade} instance. For internal use only.
     */
    @Nullable
    @Getter
    private static SecretStoreFacade secretStoreFacade = FacadeLocator.getFacade(SecretStoreFacade.class);

    /**
     * Returns the {@link SecretStoreFacade} instance.
     *
     * @throws ShouldNotHappenException
     *             If the instance is null.
     */
    private static SecretStoreFacade facade()
        throws ShouldNotHappenException
    {
        if( secretStoreFacade == null ) {
            throw new ShouldNotHappenException(
                "Failed to determine the current Cloud platform while accessing secret store information. "
                    + FacadeLocator.UNKNOWN_CLOUD_PLATFORM_MESSAGE);
        }

        return secretStoreFacade;
    }

    /**
     * Replaces the default {@link SecretStoreFacade} instance. This method is for internal use only.
     *
     * @param secretStoreFacade
     *            The facade to replace the current/default one with.
     */
    public static void setSecretStoreFacade( @Nonnull final SecretStoreFacade secretStoreFacade )
    {
        SecretStoreAccessor.secretStoreFacade = secretStoreFacade;
    }

    /**
     * Retrieves a {@link SecretStore} by its name.
     *
     * @param name
     *            The name identifying a {@link SecretStore}.
     *
     * @return The {@link SecretStore} for the given name.
     *
     * @throws SecretStoreNotFoundException
     *             If the {@link SecretStore} cannot be found.
     *
     * @throws SecretStoreAccessException
     *             If there is an issue while accessing the {@link SecretStore}.
     */
    @Nonnull
    public static SecretStore getSecretStore( final String name )
        throws SecretStoreNotFoundException,
            SecretStoreAccessException
    {
        return facade().getSecretStore(name);
    }

    /**
     * Retrieves a {@link SecretStore} by its name, if present.
     *
     * @param name
     *            The name identifying a {@link SecretStore}.
     *
     * @return An {@link Optional} of the {@link SecretStore} for the given name.
     *
     * @throws SecretStoreAccessException
     *             If there is an issue while accessing the {@link SecretStore}.
     */
    @Nonnull
    public static Optional<SecretStore> getSecretStoreIfPresent( final String name )
        throws SecretStoreAccessException
    {
        return facade().getSecretStoreIfPresent(name);
    }

    /**
     * Retrieves a {@link KeyStore} by its name.
     *
     * @param name
     *            The name identifying a {@link KeyStore}.
     *
     * @param password
     *            The password to unlock the {@link KeyStore}.
     *
     * @return The {@link KeyStore} for the given name.
     *
     * @throws KeyStoreNotFoundException
     *             If the {@link KeyStore} cannot be found.
     *
     * @throws KeyStoreAccessException
     *             If there is an issue while accessing the {@link KeyStore}.
     */
    @Nonnull
    public static KeyStore getKeyStore( final String name, final SecretStore password )
        throws KeyStoreNotFoundException,
            KeyStoreAccessException
    {
        return facade().getKeyStore(name, password);
    }

    /**
     * Retrieves a {@link KeyStore} by its name, if present.
     *
     * @param name
     *            The name identifying a {@link KeyStore}.
     *
     * @param password
     *            The password to unlock the {@link KeyStore}.
     *
     * @return The {@link KeyStore} for the given name.
     *
     * @throws KeyStoreAccessException
     *             If there is an issue while accessing the {@link KeyStore}.
     */
    @Nonnull
    public static Optional<KeyStore> getKeyStoreIfPresent( final String name, final SecretStore password )
        throws KeyStoreAccessException
    {
        return facade().getKeyStoreIfPresent(name, password);
    }
}
