/*
 * Copyright (c) 2019 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security.servlet;

import java.io.IOException;
import java.time.ZonedDateTime;
import java.util.Locale;
import java.util.Set;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.slf4j.Logger;

import com.google.common.collect.Sets;
import com.sap.cloud.sdk.cloudplatform.logging.CloudLoggerFactory;

/**
 * Activates caching in the {@link ServletResponse}, if applicable.
 * <p>
 * Caching is applicable, if the request URI <b>does not</b> contain a certain json file. If that is the case the
 * response headers for caching are added depending on the previously set response content type. Static files such as
 * stylesheets, scripts, fonts and images are not affected by this filter and will be left with the default caching
 * behavior.
 * <p>
 * It is recommended to position this filter last in the servlet filter chain.
 */
public class HttpCachingHeaderFilter implements Filter
{
    private static final Logger logger = CloudLoggerFactory.getLogger(HttpCachingHeaderFilter.class);

    // @formatter:off
    protected static final Set<String> CACHEABLE_CONTENT_TYPES = Sets.newHashSet(
            "text/css",
            "text/javascript",
            "image/webp",
            "image/png",
            "image/jpeg",
            "image/gif",
            "image/jpg",
            "image/svg+xml",
            "application/javascript",
            "application/xml",
            "application/css",
            "application/x-font-woff",
            "application/x-font-ttf"
    );
    // @formatter:on

    @Override
    public void init( final FilterConfig filterConfig )
    {

    }

    @Override
    public void destroy()
    {

    }

    @Override
    public void doFilter( final ServletRequest request, final ServletResponse response, final FilterChain chain )
        throws IOException,
            ServletException
    {
        if( request instanceof HttpServletRequest && response instanceof HttpServletResponse ) {
            final HttpServletResponse resp = (HttpServletResponse) response;
            final String requestPath = ((HttpServletRequest) request).getRequestURI();

            setResponseHeadersByRequestPath(requestPath, resp);
        }

        chain.doFilter(request, response);
    }

    private void setResponseHeadersByRequestPath( final String requestPath, final HttpServletResponse resp )
    {
        if( requestPath.contains("sap-ui-cachebuster-info.json") ) {
            if( logger.isDebugEnabled() ) {
                logger.debug("Not adding caching headers for request path '{}'.", requestPath);
            }
        } else {
            final String contentType = resp.getContentType();
            if( logger.isDebugEnabled() ) {
                logger.debug(
                    "Adding caching headers for request path '{}' and content type '{}'.",
                    requestPath,
                    contentType);
            }

            // if content type indicates static information, set expiration of cache to one day
            if( contentType != null && CACHEABLE_CONTENT_TYPES.contains(contentType.toLowerCase(Locale.ENGLISH)) ) {
                final ZonedDateTime dt = ZonedDateTime.now().plusDays(1);
                resp.setDateHeader("Expires", dt.toInstant().toEpochMilli());
            }
            // otherwise make the response non-cacheable
            else {
                resp.setHeader("Cache-Control", "no-cache, no-store");
                resp.setHeader("Expires", "0");
            }
        }
    }
}
