/*
 * Copyright (c) 2019 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security.user;

import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.common.collect.Sets;
import com.sap.cloud.sdk.cloudplatform.security.Authorization;

import lombok.Data;
import lombok.Getter;
import lombok.RequiredArgsConstructor;

@RequiredArgsConstructor
@Data
public abstract class AbstractUser implements User
{
    /**
     * {@inheritDoc}
     */
    @Getter
    @Nonnull
    private final String name;

    @Nullable
    private final Locale locale;

    @Nonnull
    private final Set<Authorization> authorizations;

    @Nonnull
    private final Map<String, UserAttribute> attributes;

    /**
     * {@inheritDoc}
     */
    @Nonnull
    @Override
    public Optional<Locale> getLocale()
    {
        return Optional.ofNullable(locale);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean hasAuthorization( @Nonnull final Authorization authorization )
    {
        return authorizations.contains(authorization);
    }

    /**
     * {@inheritDoc}
     */
    @Nonnull
    @Override
    public Set<Authorization> getAuthorizations()
    {
        return Sets.newHashSet(authorizations);
    }

    /**
     * {@inheritDoc}
     */
    @Nonnull
    @Override
    public Optional<UserAttribute> getAttribute( @Nonnull final String name )
    {
        return Optional.ofNullable(attributes.get(name));
    }
}
