/*
 * Copyright (c) 2019 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security.user;

import java.util.Optional;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;

import com.sap.cloud.sdk.cloudplatform.logging.CloudLoggerFactory;
import com.sap.cloud.sdk.cloudplatform.security.user.exception.UserAccessException;
import com.sap.cloud.sdk.cloudplatform.security.user.exception.UserNotAuthenticatedException;
import com.sap.cloud.sdk.cloudplatform.servlet.Property;
import com.sap.cloud.sdk.cloudplatform.servlet.RequestContext;
import com.sap.cloud.sdk.cloudplatform.servlet.RequestContextAccessor;
import com.sap.cloud.sdk.cloudplatform.servlet.RequestContextExecutor;
import com.sap.cloud.sdk.cloudplatform.servlet.RequestContextServletFilter;
import com.sap.cloud.sdk.cloudplatform.servlet.exception.RequestContextPropertyException;

import io.vavr.control.Try;

/**
 * Abstract base class for {@link UserFacade}s which allows to use a mocked {@link User} if the environment variable
 * "USE_MOCKED_USER" is set to "true".
 */
public abstract class AbstractUserFacade implements UserFacade
{
    private static final Logger logger = CloudLoggerFactory.getLogger(AbstractUserFacade.class);

    private static final String VARIABLE_USE_MOCKED_USER = "USE_MOCKED_USER";

    /**
     * Instantiates a mocked {@link User}.
     *
     * @return A new mocked {@link User}.
     */
    @Nonnull
    protected abstract User newMockedUser();

    /**
     * Returns a mocked {@link User}, if the environment variable "USE_MOCKED_USER" is set to "true".
     *
     * @return The mocked {@link User}, if present.
     */
    @Nonnull
    public Optional<User> getMockedUser()
    {
        @Nullable
        final String env = System.getenv(VARIABLE_USE_MOCKED_USER);

        if( Boolean.valueOf(env) ) {
            logger.error(
                "Using a mocked user with blank user name and no authorizations "
                    + "since environment variable '"
                    + VARIABLE_USE_MOCKED_USER
                    + "' is set to 'true'. "
                    + "SECURITY WARNING: This variable must never be used in productive environments!");

            return Optional.of(newMockedUser());
        }

        return Optional.empty();
    }

    /**
     * {@inheritDoc}
     */
    @Nonnull
    @Override
    public User getCurrentUser()
        throws UserNotAuthenticatedException,
            UserAccessException
    {
        final Optional<User> currentUser = getCurrentUserIfAuthenticated();

        if( !currentUser.isPresent() ) {
            throw new UserNotAuthenticatedException(
                "Failed to get current user: user not authenticated. "
                    + "For details on the security configuration, "
                    + "please refer to the SAP Cloud Platform documentation. "
                    + "Tutorials on the configuration are available at "
                    + "'https://help.sap.com/viewer/p/SAP_CLOUD_SDK'.");
        }

        return currentUser.get();
    }

    /**
     * {@inheritDoc}
     */
    @Nonnull
    @Override
    public Optional<User> getCurrentUserIfAuthenticated()
        throws UserAccessException
    {
        final Optional<RequestContext> requestContext = RequestContextAccessor.getCurrentRequestContext();

        if( !requestContext.isPresent() ) {
            throw new UserAccessException(
                "Failed to get current user: no "
                    + RequestContext.class.getSimpleName()
                    + " available."
                    + " Have you correctly configured a "
                    + RequestContextServletFilter.class.getSimpleName()
                    + " or have you wrapped your logic in a "
                    + RequestContextExecutor.class.getSimpleName()
                    + " when executing background tasks that are not triggered by a request?");
        }

        final Optional<Property<?>> property;
        try {
            property = requestContext.get().getProperty(UserRequestContextListener.PROPERTY_USER);
        }
        catch( final RequestContextPropertyException e ) {
            throw new UserAccessException(
                "Failed to get current user: failed to get " + RequestContext.class.getSimpleName() + " property.",
                e);
        }

        if( !property.isPresent() ) {
            throw new UserAccessException(
                "Failed to get current tenant: "
                    + RequestContext.class.getSimpleName()
                    + " property '"
                    + UserRequestContextListener.PROPERTY_USER
                    + "' not initialized."
                    + " Have you correctly configured a "
                    + UserRequestContextListener.class.getSimpleName()
                    + " in the relevant "
                    + RequestContextServletFilter.class.getSimpleName()
                    + " or "
                    + RequestContextExecutor.class.getSimpleName()
                    + "?");
        }

        @Nullable
        final Exception exception = property.get().getException();
        if( exception != null ) {
            throw new UserAccessException("Failed to get current user.", exception);
        }

        @Nullable
        final User user = (User) property.get().getValue();
        return Optional.ofNullable(user);
    }

    /**
     * {@inheritDoc}
     */
    @Nonnull
    @Override
    public Try<User> tryGetCurrentUser()
    {
        return Try.of(this::getCurrentUser);
    }
}
