/*
 * Copyright (c) 2019 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security.user;

import java.util.Optional;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.sap.cloud.sdk.cloudplatform.security.user.exception.UserAccessDeniedException;
import com.sap.cloud.sdk.cloudplatform.security.user.exception.UserAccessException;
import com.sap.cloud.sdk.cloudplatform.security.user.exception.UserNotAuthenticatedException;

import io.vavr.control.Try;

/**
 * Facade interface encapsulating the logic to access {@link User} information.
 */
public interface UserFacade
{
    /**
     * Returns the platform-specific implementation class of {@link User}. For internal use only.
     *
     * @return The type of the platform-specific implementation.
     */
    @Nonnull
    Class<? extends User> getUserClass();

    /**
     * Resolves the current {@link User} using the platform-specific lookup. For internal use only.
     * <p>
     * <strong>Caution: This method must only be invoked within the container-managed thread of a request.</strong>
     *
     * @return An {@link Optional} of the current {@link User}.
     *
     * @throws UserAccessException
     *             If there is an issue while resolving the {@link User}.
     *
     * @see UserRequestContextListener
     */
    @Nonnull
    Optional<User> resolveCurrentUser()
        throws UserAccessException;

    /**
     * Returns the current {@link User}.
     *
     * @return The current {@link User}.
     *
     * @throws UserNotAuthenticatedException
     *             If the {@link User} is currently not authenticated. This typically occurs when trying to access the
     *             user within code that is running outside the context of a request, e.g., within a background task.
     *             For more details on how user authentication is defined, refer to
     *             {@link #getCurrentUserIfAuthenticated()}.
     *
     * @throws UserAccessException
     *             If there is an issue while accessing the {@link User}.
     */
    @Nonnull
    User getCurrentUser()
        throws UserNotAuthenticatedException,
            UserAccessException;

    /**
     * Returns the current {@link User}, if authenticated.
     * <p>
     * The user authentication is defined as follows:
     * <table border="1">
     * <tr>
     * <th></th>
     * <th>User authenticated</th>
     * <th>User not authenticated</th>
     * </tr>
     * <tr>
     * <td><strong>SAP Cloud Platform Cloud Foundry</strong></td>
     * <td>A request is present with an "Authorization" header that contains a valid JWT bearer with field
     * "user_name".</td>
     * <td>A request is not available, no "Authorization" header is present in the current request, or the JWT bearer
     * does not hold a field "user_name".</td>
     * </tr>
     * <tr>
     * <td><strong>SAP Cloud Platform Neo</strong></td>
     * <td>A request is present that holds an authenticated principal according to the {@code UserProvider} API.</td>
     * <td>A request is not available or the user is not authenticated according to the {@code UserProvider} API.</td>
     * </tr>
     * </table>
     *
     * @see <a href="https://help.hana.ondemand.com/javadoc/com/sap/security/um/user/UserProvider.html"> SAP Cloud
     *      Platform Neo - UserProvider API </a>
     *
     * @return An {@link Optional} of the current {@link User}.
     *
     * @throws UserAccessException
     *             If there is an issue while accessing the {@link User}.
     */
    @Nonnull
    Optional<User> getCurrentUserIfAuthenticated()
        throws UserAccessException;

    /**
     * Returns a {@link Try} for the current {@link User}.
     *
     * @return A {@link Try} for the current {@link User}.
     */
    @Nonnull
    Try<User> tryGetCurrentUser();

    /**
     * Returns a {@link User} by its name, if it exists.
     *
     * @param name
     *            The name of the {@link User} to find.
     *
     * @return An optional, wrapping the user with the given name.
     *
     * @throws UserAccessDeniedException
     *             If accessing user information is denied.
     *
     * @throws UserAccessException
     *             If there is an issue while accessing the {@link User}.
     */
    @Nonnull
    Optional<User> getUserByName( @Nullable final String name )
        throws UserAccessDeniedException,
            UserAccessException;
}
