/*
 * Copyright (c) 2020 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security.servlet;

import java.io.IOException;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * Adjusts the headers of {@link HttpServletRequest} and {@link HttpServletResponse}.
 * <p>
 * The response header gets the {@code Access-Control-Allow-Origin} value set to {@code *}. If the request method is
 * {@code OPTIONS}, the response header also gets {@code Access-Control-Allow-Methods} set to {@code GET} and
 * {@code Access-Control-Allow-Headers} set to {@code accept, authorization}. Modified requests and responses are then
 * rejected by the filter.
 */
public class HttpAllowOriginHeaderFilter implements Filter
{
    @Override
    public void init( final FilterConfig filterConfig )
        throws ServletException
    {

    }

    @Override
    public void doFilter( final ServletRequest request, final ServletResponse response, final FilterChain chain )
        throws IOException,
            ServletException
    {
        if( request instanceof HttpServletRequest && response instanceof HttpServletResponse ) {
            final HttpServletResponse resp = (HttpServletResponse) response;
            final HttpServletRequest req = (HttpServletRequest) request;

            resp.setHeader("Access-Control-Allow-Origin", "*");

            if( req.getMethod().equals("OPTIONS") ) {
                resp.setHeader("Access-Control-Allow-Methods", "GET");
                resp.setHeader("Access-Control-Allow-Headers", "accept, authorization");
                return;
            }
        }

        chain.doFilter(request, response);
    }

    @Override
    public void destroy()
    {

    }
}
