/*
 * Copyright (c) 2020 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security.user;

import java.util.Optional;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.sap.cloud.sdk.cloudplatform.exception.ShouldNotHappenException;
import com.sap.cloud.sdk.cloudplatform.security.user.exception.UserAccessDeniedException;
import com.sap.cloud.sdk.cloudplatform.security.user.exception.UserAccessException;
import com.sap.cloud.sdk.cloudplatform.security.user.exception.UserNotAuthenticatedException;
import com.sap.cloud.sdk.cloudplatform.util.FacadeLocator;

import io.vavr.control.Try;
import lombok.Getter;

/**
 * Main class find and access {@link User}.
 * <p>
 * This class handles the access to a {@code User} by delegating the calls to the underlying {@link UserFacade}.
 */
public final class UserAccessor
{
    /**
     * Returns the {@link UserFacade} instance. For internal use only.
     */
    @Nullable
    @Getter
    private static UserFacade userFacade = FacadeLocator.getFacade(UserFacade.class);

    /**
     * Returns the {@link UserFacade} instance.
     *
     * @throws ShouldNotHappenException
     *             If the {@link UserFacade} instance is null.
     */
    @Nonnull
    static UserFacade facade()
        throws ShouldNotHappenException
    {
        if( userFacade == null ) {
            throw new ShouldNotHappenException(
                "Failed to determine the current Cloud platform while accessing user information. "
                    + FacadeLocator.UNKNOWN_CLOUD_PLATFORM_MESSAGE);
        }

        return userFacade;
    }

    /**
     * Replaces the default {@link UserFacade} instance. This method is for internal use only.
     *
     * @param userFacade
     *            The facade to replace the current/default one with.
     */
    public static void setUserFacade( @Nonnull final UserFacade userFacade )
    {
        UserAccessor.userFacade = userFacade;
    }

    /**
     * Returns the current {@link User}.
     *
     * @return The current {@link User}.
     *
     * @throws UserNotAuthenticatedException
     *             If the {@link User} is currently not authenticated. This typically occurs when trying to access the
     *             user within code that is running outside the context of a request, e.g., within a background task.
     *             For more details on how user authentication is defined, refer to
     *             {@link #getCurrentUserIfAuthenticated()}.
     *
     * @throws UserAccessException
     *             If there is an issue while accessing the {@link User}.
     */
    @Nonnull
    public static User getCurrentUser()
        throws UserNotAuthenticatedException,
            UserAccessException
    {
        return facade().getCurrentUser();
    }

    /**
     * Returns the current {@link User}, if authenticated.
     * <p>
     * The user authentication is defined as follows:
     * <table border="1">
     * <tr>
     * <th></th>
     * <th>User authenticated</th>
     * <th>User not authenticated</th>
     * </tr>
     * <tr>
     * <td><strong>SAP Cloud Platform Cloud Foundry</strong></td>
     * <td>A request is present with an "Authorization" header that contains a valid JWT bearer with field
     * "user_name".</td>
     * <td>A request is not available, no "Authorization" header is present in the current request, or the JWT bearer
     * does not hold a field "user_name".</td>
     * </tr>
     * <tr>
     * <td><strong>SAP Cloud Platform Neo</strong></td>
     * <td>A request is present that holds an authenticated principal according to the {@code UserProvider} API.</td>
     * <td>A request is not available or the user is not authenticated according to the {@code UserProvider} API.</td>
     * </tr>
     * </table>
     *
     * @see <a href="https://help.hana.ondemand.com/javadoc/com/sap/security/um/user/UserProvider.html"> SAP Cloud
     *      Platform Neo - UserProvider API </a>
     *
     * @return An {@link Optional} of the current {@link User}.
     *
     * @throws UserAccessException
     *             If there is an issue while accessing the {@link User}.
     */
    @Nonnull
    public static Optional<User> getCurrentUserIfAuthenticated()
        throws UserAccessException
    {
        return facade().getCurrentUserIfAuthenticated();
    }

    /**
     * Returns a {@link Try} for the current {@link User}.
     *
     * @return A {@link Try} for the current {@link User}.
     */
    @Nonnull
    public static Try<User> tryGetCurrentUser()
    {
        return facade().tryGetCurrentUser();
    }

    /**
     * Returns a {@link User} by its name, if it exists.
     *
     * @param name
     *            The name of the {@link User} to find.
     *
     * @return An optional, wrapping the user with the given name.
     *
     * @throws UserAccessDeniedException
     *             If accessing user information is denied.
     *
     * @throws UserAccessException
     *             If there is an issue while accessing the {@link User}.
     */
    @Nonnull
    public static Optional<User> getUserByName( @Nullable final String name )
        throws UserAccessDeniedException,
            UserAccessException
    {
        return facade().getUserByName(name);
    }
}
