/*
 * Copyright (c) 2020 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security.user.exception;

import java.util.Collections;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.sap.cloud.sdk.cloudplatform.security.Role;
import com.sap.cloud.sdk.cloudplatform.security.user.User;
import com.sap.cloud.sdk.cloudplatform.servlet.response.AccessDeniedResponse;
import com.sap.cloud.sdk.cloudplatform.servlet.response.ResponseWithErrorCode;
import com.sap.cloud.sdk.cloudplatform.servlet.response.WithErrorResponse;

import lombok.Getter;

/**
 * Exception indicating that a user does not have a required role.
 */
public class UserNotInRoleException extends RuntimeException implements WithErrorResponse
{
    private static final long serialVersionUID = 3870535945565254939L;

    @Getter
    @Nullable
    private final transient User user;

    @Getter
    @Nullable
    private final transient Role role;

    /**
     * Creates the exception with an error message based on the given user and role.
     * 
     * @param user
     *            The user which didn't have the needed role.
     * @param role
     *            The role which the user didn't have.
     */
    public UserNotInRoleException( @Nullable final User user, @Nullable final Role role )
    {
        super(
            "User "
                + (user != null ? user.getName() + " " : "")
                + "does not have role"
                + (role != null ? " " + role.getName() : "")
                + ".");

        this.user = user;
        this.role = role;
    }

    /**
     * Creates the exception by delegating the message to the super constructor and storing the user and the rule.
     * 
     * @param user
     *            The user which didn't have the needed role.
     * @param role
     *            The role which the user didn't have.
     * @param message
     *            The exception message.
     */
    public UserNotInRoleException( @Nullable final User user, @Nullable final Role role, final String message )
    {
        super(message);

        this.user = user;
        this.role = role;
    }

    /**
     * Creates the exception by delegating the message and the causing exception to the super constructor and storing
     * the user and the rule.
     * 
     * @param user
     *            The user which didn't have the needed role.
     * @param role
     *            The role which the user didn't have.
     * @param message
     *            The exception message.
     * @param cause
     *            The exception that caused the exception to be created.
     */
    public UserNotInRoleException(
        @Nullable final User user,
        @Nullable final Role role,
        final String message,
        final Throwable cause )
    {
        super(message, cause);

        this.user = user;
        this.role = role;
    }

    @Nonnull
    @Override
    public ResponseWithErrorCode getErrorResponse()
    {
        return new AccessDeniedResponse(
            user != null ? user.getName() : null,
            role != null ? Collections.singleton(role.getName()) : null,
            getMessage());
    }
}
