/*
 * Copyright (c) 2020 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security.secret;

import java.security.KeyStore;
import java.util.Optional;

import javax.annotation.Nonnull;

import com.sap.cloud.sdk.cloudplatform.security.secret.exception.KeyStoreAccessException;
import com.sap.cloud.sdk.cloudplatform.security.secret.exception.KeyStoreNotFoundException;
import com.sap.cloud.sdk.cloudplatform.security.secret.exception.SecretStoreAccessException;
import com.sap.cloud.sdk.cloudplatform.security.secret.exception.SecretStoreNotFoundException;

/**
 * Facade interface encapsulating the access to concrete {@link SecretStore} and {@link KeyStore} classes.
 */
public interface SecretStoreFacade
{
    /**
     * Returns the platform-specific implementation class of {@link SecretStore}. For internal use only.
     *
     * @return The type of the platform-specific implementation.
     */
    @Nonnull
    Class<? extends SecretStore> getSecretStoreClass();

    /**
     * Retrieves a {@link SecretStore} by its name.
     *
     * @param name
     *            The name identifying a {@link SecretStore}.
     *
     * @return The {@link SecretStore} for the given name.
     *
     * @throws SecretStoreNotFoundException
     *             If the {@link SecretStore} cannot be found.
     *
     * @throws SecretStoreAccessException
     *             If there is an issue while accessing the {@link SecretStore}.
     */
    @Nonnull
    SecretStore getSecretStore( final String name )
        throws SecretStoreNotFoundException,
            SecretStoreAccessException;

    /**
     * Retrieves a {@link SecretStore} by its name, if present.
     *
     * @param name
     *            The name identifying a {@link SecretStore}.
     *
     * @return An {@link Optional} of the {@link SecretStore} for the given name.
     *
     * @throws SecretStoreAccessException
     *             If there is an issue while accessing the {@link SecretStore}.
     */
    @Nonnull
    Optional<SecretStore> getSecretStoreIfPresent( final String name )
        throws SecretStoreAccessException;

    /**
     * Retrieves a {@link KeyStore} by its name.
     *
     * @param name
     *            The name identifying a {@link KeyStore}.
     *
     * @param password
     *            The password to unlock the {@link KeyStore}.
     *
     * @return The {@link KeyStore} for the given name.
     *
     * @throws KeyStoreNotFoundException
     *             If the {@link KeyStore} cannot be found.
     *
     * @throws KeyStoreAccessException
     *             If there is an issue while accessing the {@link KeyStore}.
     */
    @Nonnull
    KeyStore getKeyStore( final String name, final SecretStore password )
        throws KeyStoreNotFoundException,
            KeyStoreAccessException;

    /**
     * Retrieves a {@link KeyStore} by its name, if present.
     *
     * @param name
     *            The name identifying a {@link KeyStore}.
     *
     * @param password
     *            The password to unlock the {@link KeyStore}.
     *
     * @return The {@link KeyStore} for the given name.
     *
     * @throws KeyStoreAccessException
     *             If there is an issue while accessing the {@link KeyStore}.
     */
    @Nonnull
    Optional<KeyStore> getKeyStoreIfPresent( final String name, final SecretStore password )
        throws KeyStoreAccessException;
}
