/*
 * Copyright (c) 2020 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.typeconverter;

import java.lang.reflect.Type;

import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonParseException;

/**
 * Implementation of {@code JsonDeserializer} deserializing a {@code JsonPrimitive} to an object of type {@code T} using
 * the given {@code TypeConverter}.
 *
 * @param <T>
 *            The type of the object created by deserialization.
 */
public class TypeConverterGsonDeserializer<T> implements JsonDeserializer<T>
{
    private final TypeConverter<T, String> typeConverter;

    /**
     * Creates a deserializer with the given String-converter.
     * 
     * @param typeConverter
     *            The converter to deserialize the JSON String.
     * @param <ConverterT>
     *            The type of the converter to be used.
     */
    public <ConverterT extends TypeConverter<T, String>> TypeConverterGsonDeserializer( final ConverterT typeConverter )
    {
        this.typeConverter = typeConverter;
    }

    @Override
    public T deserialize( final JsonElement json, final Type typeOfT, final JsonDeserializationContext context )
        throws JsonParseException
    {
        return typeConverter.fromDomain(json.getAsJsonPrimitive().getAsString()).orNull();
    }
}
