/*
 * Copyright (c) 2020 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.auditlog;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.servlet.http.HttpServletRequest;

import com.sap.cloud.sdk.cloudplatform.security.principal.Principal;
import com.sap.cloud.sdk.cloudplatform.security.principal.PrincipalAccessor;
import com.sap.cloud.sdk.cloudplatform.servlet.RequestAccessor;
import com.sap.cloud.sdk.cloudplatform.tenant.Tenant;
import com.sap.cloud.sdk.cloudplatform.tenant.TenantAccessor;

import io.vavr.control.Option;
import io.vavr.control.Try;
import lombok.AllArgsConstructor;
import lombok.Data;

/**
 * Represents the requester of an access to an object in the context of an audit log.
 * <p>
 * Therefore this class stores identifiable information of the requester. Information about the field accessed as well
 * as about the access time are handled in the implementation of the {@link AuditLog} interface.
 */
@Data
@AllArgsConstructor
public class AccessRequester
{
    /**
     * Field representing an unknown requester.
     */
    public static final AccessRequester UNKNOWN = new AccessRequester(null, null, null, null);

    /**
     * Default user Id, used in case of an unknown remote user.
     */
    public static final String UNAUTHENTICATED_USER = "noauth";

    /**
     * Tenant Id of the user requesting the access.
     */
    @Nullable
    private final String tenantId;

    /**
     * Id of the user requesting the access.
     */
    @Nullable
    private final String principalId;

    /**
     * The IP address of the user requesting the access.
     */
    @Nullable
    private final String ipAddress;

    /**
     * The connection channel used, e.g. http, or https.
     */
    @Nullable
    private final String channel;

    /**
     * The Id of the remote user requesting access.
     * 
     * @return An {@link Option} containing the user Id.
     */
    @Nonnull
    public Option<String> getPrincipalId()
    {
        return Option.of(principalId);
    }

    /**
     * The tenant Id of the remote user requesting access.
     *
     * @return An {@link Option} containing the tenant Id.
     */
    @Nonnull
    public Option<String> getTenantId()
    {
        return Option.of(tenantId);
    }

    /**
     * The IP address of the remote user requesting access.
     * 
     * @return An {@link Option} containing the IP address.
     */
    @Nonnull
    public Option<String> getIpAddress()
    {
        return Option.of(ipAddress);
    }

    /**
     * The connection channel used, e.g. http, or https.
     * 
     * @return An {@link Option} containing the connection channel.
     */
    @Nonnull
    public Option<String> getChannel()
    {
        return Option.of(channel);
    }

    /**
     * Creates an AccessRequester based on the given request.
     * 
     * @param request
     *            The request which tries to access an object.
     */
    public AccessRequester( @Nonnull final HttpServletRequest request )
    {
        principalId = getCurrentPrincipalId();
        tenantId = null;
        ipAddress = request.getRemoteAddr();
        channel = request.getScheme();
    }

    /**
     * Creates an AccessRequester based on the given request and tenant.
     *
     * @param request
     *            The request which tries to access an object.
     * @param tenant
     *            The tenant which tries to access an object.
     */
    public AccessRequester( @Nonnull final HttpServletRequest request, @Nonnull final Tenant tenant )
    {
        principalId = getCurrentPrincipalId();
        tenantId = tenant.getTenantId();
        ipAddress = request.getRemoteAddr();
        channel = request.getScheme();
    }

    /**
     * Creates a request based on the given, nullable request.
     * 
     * @param request
     *            The request which tries to access an object.
     *
     * @return An AccessRequester for the given request. Never {@code null}.
     */
    @Nonnull
    public static AccessRequester of( @Nullable final HttpServletRequest request )
    {
        if( request == null ) {
            return AccessRequester.UNKNOWN;
        }
        return new AccessRequester(request);
    }

    /**
     * Creates a request based on the given, nullable request and tenant.
     *
     * @param request
     *            The request which tries to access an object.
     * @param tenant
     *            The tenant which tries to access an object.
     *
     * @return An AccessRequester for the given request. Never {@code null}.
     */
    @Nonnull
    public static AccessRequester of( @Nullable final HttpServletRequest request, @Nullable final Tenant tenant )
    {
        if( request != null ) {
            if( tenant != null ) {
                return new AccessRequester(request, tenant);
            } else {
                return new AccessRequester(request);
            }
        } else {
            return AccessRequester.UNKNOWN;
        }
    }

    /**
     * Creates a request based on the current request.
     * 
     * @return An AccessRequester for the current request. Never {@code null}.
     */
    @Nonnull
    public static AccessRequester ofCurrentRequest()
    {
        @Nullable
        final HttpServletRequest request = RequestAccessor.tryGetCurrentRequest().getOrNull();

        final Try<Tenant> tenantTry = TenantAccessor.tryGetCurrentTenant();

        if( tenantTry.isSuccess() ) {
            return of(request, tenantTry.get());
        } else {
            return of(request);
        }
    }

    @Nonnull
    private String getCurrentPrincipalId()
    {
        return PrincipalAccessor.tryGetCurrentPrincipal().map(Principal::getPrincipalId).getOrElse(
            UNAUTHENTICATED_USER);
    }
}
