/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.auditlog;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.sap.cloud.sdk.cloudplatform.auditlog.exception.AuditLogAccessException;

/**
 * Interface for audit logging.
 * <p>
 * Important: For performance reasons, use only efficiently for logging of relevant events. The logging methods allow to
 * log:
 * <ul>
 * <li>security relevant events,
 * <li>read access to sensitive personal data,
 * <li>changes to configuration data,
 * <li>and changes to personal data.
 * </ul>
 */
public interface AuditLog
{
    /**
     * Logs the start of a generic security event.
     *
     * @param initiator
     *            Information about the user that starts performing the security event.
     * @param message
     *            Descriptive log message.
     *
     * @throws AuditLogAccessException
     *             If there is an issue while accessing the {@link AuditLog}.
     */
    void logSecurityEventBeginning( @Nonnull final AccessRequester initiator, @Nullable final String message )
        throws AuditLogAccessException;

    /**
     * Logs a generic security event.
     *
     * @param initiator
     *            Information about the user that performed the security event.
     * @param message
     *            Descriptive log message.
     * @param throwable
     *            The exception in case of an error.
     *
     * @throws AuditLogAccessException
     *             If there is an issue while accessing the {@link AuditLog}.
     */
    void logSecurityEvent(
        @Nonnull final AccessRequester initiator,
        @Nullable final String message,
        @Nullable final Throwable throwable )
        throws AuditLogAccessException;

    /**
     * Logs the start of a change of some attributes of configuration data.
     *
     * @param initiator
     *            Information about the user starting to modify the configuration.
     * @param object
     *            An instance of {@link AuditedDataObject} that represents the object that will be modified.
     * @param attributeAffected
     *            An attribute that will be changed.
     * @param attributesAffected
     *            An arbitrary number of attributes that will be changed.
     *
     * @throws AuditLogAccessException
     *             If there is an issue while accessing the {@link AuditLog}.
     */
    void logConfigChangeBeginning(
        @Nonnull final AccessRequester initiator,
        @Nonnull final AuditedDataObject object,
        @Nonnull final AccessedAttribute attributeAffected,
        @Nullable final AccessedAttribute... attributesAffected )
        throws AuditLogAccessException;

    /**
     * Logs the change of some attributes of configuration data.
     *
     * @param initiator
     *            Information about the user modifying the configuration.
     * @param object
     *            An instance of {@link AuditedDataObject} that represents the object that has been modified.
     * @param error
     *            The exception in case of an error.
     * @param attributeAffected
     *            An attribute that has been changed.
     * @param attributesAffected
     *            An arbitrary number of attributes that have been changed.
     *
     * @throws AuditLogAccessException
     *             If there is an issue while accessing the {@link AuditLog}.
     */
    void logConfigChange(
        @Nonnull final AccessRequester initiator,
        @Nonnull final AuditedDataObject object,
        @Nullable final Throwable error,
        @Nonnull final AccessedAttribute attributeAffected,
        @Nullable final AccessedAttribute... attributesAffected )
        throws AuditLogAccessException;

    /**
     * Logs the <b>attempt</b> to read some attributes from an object.
     *
     * @param initiator
     *            Information about the user trying to the read the attributes.
     * @param object
     *            An instance of {@link AuditedDataObject} that represents the object to be read.
     * @param subject
     *            An instance of {@link AuditedDataSubject} that represents the owner of the data to be read.
     * @param attributeAffected
     *            An attribute that should be read.
     * @param attributesAffected
     *            An arbitrary number of attributes that should be read.
     *
     * @throws AuditLogAccessException
     *             If there is an issue while accessing the {@link AuditLog}.
     */
    void logDataReadAttempt(
        @Nonnull final AccessRequester initiator,
        @Nonnull final AuditedDataObject object,
        @Nonnull final AuditedDataSubject subject,
        @Nonnull final AccessedAttribute attributeAffected,
        @Nullable final AccessedAttribute... attributesAffected )
        throws AuditLogAccessException;

    /**
     * Logs the read access to some attributes of an object.
     *
     * @param initiator
     *            Information about the user reading the attributes.
     * @param object
     *            An instance of {@link AuditedDataObject} that represents the object that has been read.
     * @param subject
     *            An instance of {@link AuditedDataSubject} that represents the owner of the data that has been read.
     * @param error
     *            The exception in case of an error.
     * @param attributeAffected
     *            An attribute that has been read.
     * @param attributesAffected
     *            An arbitrary number of attributes that have been read.
     *
     * @throws AuditLogAccessException
     *             If there is an issue while accessing the {@link AuditLog}.
     */
    void logDataRead(
        @Nonnull final AccessRequester initiator,
        @Nonnull final AuditedDataObject object,
        @Nonnull final AuditedDataSubject subject,
        @Nullable final Throwable error,
        @Nonnull final AccessedAttribute attributeAffected,
        @Nullable final AccessedAttribute... attributesAffected )
        throws AuditLogAccessException;

    /**
     * Logs the attempt to write to some attributes of an object.
     *
     * @param initiator
     *            Information about the user trying to write to some attributes.
     * @param object
     *            An instance of {@link AuditedDataObject} that represents the object that is about to be modified.
     * @param subject
     *            An instance of {@link AuditedDataSubject} that represents the owner of the data that is about to be
     *            modified.
     * @param attributeAffected
     *            An attribute that should be modified, containing the old and new value.
     * @param attributesAffected
     *            An arbitrary number of attributes that should be modified, containing the old and new values.
     *
     * @throws AuditLogAccessException
     *             If there is an issue while accessing the {@link AuditLog}.
     */
    void logDataWriteAttempt(
        @Nonnull final AccessRequester initiator,
        @Nonnull final AuditedDataObject object,
        @Nonnull final AuditedDataSubject subject,
        @Nonnull final AccessedAttribute attributeAffected,
        @Nullable final AccessedAttribute... attributesAffected )
        throws AuditLogAccessException;

    /**
     * Logs the modification of some attributes of an object.
     *
     * @param initiator
     *            Information about the user changing the attributes.
     * @param object
     *            An instance of {@link AuditedDataObject} that represents the object that has been modified.
     * @param subject
     *            An instance of {@link AuditedDataSubject} that represents the owner of the data that has been
     *            modified.
     * @param error
     *            The exception in case of an error.
     * @param attributeAffected
     *            An attribute that has been modified, containing the old and new value.
     * @param attributesAffected
     *            An arbitrary number of attributes that have been modified, containing the old and new values.
     *
     * @throws AuditLogAccessException
     *             If there is an issue while accessing the {@link AuditLog}.
     */
    void logDataWrite(
        @Nonnull final AccessRequester initiator,
        @Nonnull final AuditedDataObject object,
        @Nonnull final AuditedDataSubject subject,
        @Nullable final Throwable error,
        @Nonnull final AccessedAttribute attributeAffected,
        @Nullable final AccessedAttribute... attributesAffected )
        throws AuditLogAccessException;
}
