/*
 * Copyright (c) 2022 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.auditlog;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import lombok.Data;

/**
 * This class represents the access to a security relevant attribute in the context of an audit log.
 * <p>
 * Therefore this class stores which field was accessed in which way. Information about the initiator of the access as
 * well as about the access time are handled in the implementation of the {@link AuditLog} interface.
 */
@Data
public class AccessedAttribute
{
    /** Enum representing the type of operation performed on the accessed attribute. */
    public enum Operation
    {
        /** Indicates that the attribute was initiated with some data. */
        CREATE,

        /** Indicates that the attribute was finally cleared. */
        DELETE,

        /** Indicates that the attribute was read, without any modification. */
        READ,

        /** Indicates that the attribute was modified. */
        WRITE
    }

    /** An unique identifier of the attribute accessed. */
    public final String identifier;

    /**
     * The type of operation performed on the attribute. Not logged on Cloud Foundry.
     */
    public final Operation operation;

    /**
     * A human readable representation of the attribute.
     */
    @Nullable
    public final String displayName;

    /**
     * The value of the attribute prior to the access.
     */
    @Nullable
    public final Object oldValue;

    /**
     * The value of the attribute after the access.
     */
    @Nullable
    public final Object newValue;

    /**
     * Flag indicating whether the operation succeeded.
     */
    public final boolean operationSuccessful;

    /**
     * Creates an AccessedAttribute instance based on all fields.
     *
     * @param identifier
     *            An identifier for the attribute accessed.
     * @param operation
     *            The type of operation performed on the attribute. Not logged on Cloud Foundry.
     * @param displayName
     *            Human readable representation of the attribute.
     * @param oldValue
     *            The value of the attribute prior to the access.
     * @param newValue
     *            The value of the attribute after the access.
     * @param operationSuccessful
     *            Flag indicating whether the operation succeeded.
     */
    public AccessedAttribute(
        @Nonnull final String identifier,
        @Nonnull final Operation operation,
        @Nullable final String displayName,
        @Nullable final Object oldValue,
        @Nullable final Object newValue,
        final boolean operationSuccessful )
    {
        this.identifier = identifier;
        this.operation = operation;
        this.displayName = displayName;
        this.oldValue = oldValue;
        this.newValue = newValue;
        this.operationSuccessful = operationSuccessful;
    }

    /**
     * Creates an AccessedAttribute with a successful operation.
     *
     * @param identifier
     *            An identifier for the attribute accessed.
     * @param operation
     *            The type of operation performed on the attribute. Not logged on Cloud Foundry.
     * @param displayName
     *            Human readable representation of the attribute.
     * @param oldValue
     *            The value of the attribute prior to the access.
     * @param newValue
     *            The value of the attribute after the access.
     */
    public AccessedAttribute(
        @Nonnull final String identifier,
        @Nonnull final Operation operation,
        @Nullable final String displayName,
        @Nullable final Object oldValue,
        @Nullable final Object newValue )
    {
        this(identifier, operation, displayName, oldValue, newValue, true);
    }

    /**
     * Creates an AccessedAttribute based on the mandatory fields.
     * 
     * @param identifier
     *            An identifier for the attribute accessed.
     * @param operation
     *            The type of operation performed on the attribute. Not logged on Cloud Foundry.
     */
    public AccessedAttribute( @Nonnull final String identifier, @Nonnull final Operation operation )
    {
        this(identifier, operation, null, null, null, true);
    }
}
