/*
 * Copyright (c) 2022 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.auditlog;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.sap.cloud.sdk.cloudplatform.auditlog.exception.AuditLogAccessException;
import com.sap.cloud.sdk.cloudplatform.util.FacadeLocator;

import io.vavr.control.Try;

/**
 * Main class of the audit log functionality.
 * <p>
 * This class handles all logging request by delegating the calls to the {@link AuditLog} implementation given by the
 * {@link AuditLogFacade}.
 */
public final class AuditLogger
{
    @Nonnull
    private static Try<AuditLogFacade> auditLogFacade = FacadeLocator.getFacade(AuditLogFacade.class);

    /**
     * Returns the {@link AuditLogFacade} instance. For internal use only.
     *
     * @return The {@link AuditLogFacade} instance, or {@code null}.
     */
    @Nullable
    public static AuditLogFacade getAuditLogFacade()
    {
        return auditLogFacade.getOrNull();
    }

    /**
     * Returns a {@link Try} of the {@link AuditLogFacade} instance. For internal use only.
     *
     * @return A {@link Try} of the {@link AuditLogFacade} instance.
     */
    @Nonnull
    public static Try<AuditLogFacade> tryGetAuditLogFacade()
    {
        return auditLogFacade;
    }

    /**
     * Replaces the default {@link AuditLogFacade} instance. This method is for internal use only.
     *
     * @param auditLogFacade
     *            An instance of {@link AuditLogFacade}.
     */
    public static void setAuditLogFacade( @Nonnull final AuditLogFacade auditLogFacade )
    {
        AuditLogger.auditLogFacade = Try.success(auditLogFacade);
    }

    /**
     * Returns the {@link AuditLog}.
     *
     * @return The {@link AuditLog}.
     *
     * @throws AuditLogAccessException
     *             If there is an issue while accessing the {@link AuditLog}.
     */
    @Nonnull
    public static AuditLog getAuditLog()
        throws AuditLogAccessException
    {
        return tryGetAuditLog().getOrElseThrow(failure -> {
            if( failure instanceof AuditLogAccessException ) {
                throw (AuditLogAccessException) failure;
            } else {
                throw new AuditLogAccessException("Failed to get audit log.", failure);
            }
        });
    }

    /**
     * Returns a {@link Try} of the {@link AuditLog}.
     *
     * @return A {@link Try} of the {@link AuditLog}.
     */
    @Nonnull
    public static Try<AuditLog> tryGetAuditLog()
    {
        return auditLogFacade.flatMap(AuditLogFacade::tryGetAuditLog);
    }

    /**
     * Logs the start of a generic security event.
     *
     * @param initiator
     *            Information about the user that starts performing the security event.
     * @param message
     *            Descriptive log message.
     *
     * @throws AuditLogAccessException
     *             If there is an issue while accessing the {@link AuditLog}.
     */
    public static
        void
        logSecurityEventBeginning( @Nonnull final AccessRequester initiator, @Nullable final String message )
            throws AuditLogAccessException
    {
        getAuditLog().logSecurityEventBeginning(initiator, message);
    }

    /**
     * Logs the start of a generic security event by the current request.
     *
     * @param message
     *            Descriptive log message.
     */
    public static void logSecurityEventBeginning( @Nullable final String message )
    {
        logSecurityEventBeginning(AccessRequester.ofCurrentRequest(), message);
    }

    /**
     * Logs a generic security event.
     *
     * @param initiator
     *            Information about the user that performed the security event.
     * @param message
     *            Descriptive log message.
     * @param throwable
     *            The exception in case of an error.
     */
    public static void logSecurityEvent(
        @Nonnull final AccessRequester initiator,
        @Nullable final String message,
        @Nullable final Throwable throwable )
    {
        getAuditLog().logSecurityEvent(initiator, message, throwable);
    }

    /**
     * Logs a generic security event by the current request.
     *
     * @param message
     *            Descriptive log message.
     * @param throwable
     *            The exception in case of an error.
     */
    public static void logSecurityEvent( @Nullable final String message, @Nullable final Throwable throwable )
    {
        logSecurityEvent(AccessRequester.ofCurrentRequest(), message, throwable);
    }

    /**
     * Logs the start of a change of some attributes of configuration data.
     *
     * @param initiator
     *            Information about the user starting to modify the configuration.
     * @param object
     *            An instance of {@link AuditedDataObject} that represents the object that will be modified.
     * @param attributeAffected
     *            An attribute that will be changed.
     * @param attributesAffected
     *            An arbitrary number of attributes that will be changed.
     */
    public static void logConfigChangeBeginning(
        @Nonnull final AccessRequester initiator,
        @Nonnull final AuditedDataObject object,
        @Nonnull final AccessedAttribute attributeAffected,
        @Nullable final AccessedAttribute... attributesAffected )
    {
        getAuditLog().logConfigChangeBeginning(initiator, object, attributeAffected, attributesAffected);
    }

    /**
     * Logs the start of a change of some attributes of configuration data by the current request.
     *
     * @param object
     *            An instance of {@link AuditedDataObject} that represents the object that will be modified.
     * @param attributeAffected
     *            An attribute that will be changed.
     * @param attributesAffected
     *            An arbitrary number of attributes that will be changed.
     */
    public static void logConfigChangeBeginning(
        @Nonnull final AuditedDataObject object,
        @Nonnull final AccessedAttribute attributeAffected,
        @Nullable final AccessedAttribute... attributesAffected )
    {
        logConfigChangeBeginning(AccessRequester.ofCurrentRequest(), object, attributeAffected, attributesAffected);
    }

    /**
     * Logs the change of some attributes of configuration data.
     *
     * @param initiator
     *            Information about the user modifying the configuration.
     * @param object
     *            An instance of {@link AuditedDataObject} that represents the object that has been modified.
     * @param error
     *            The exception in case of an error.
     * @param attributeAffected
     *            An attribute that has been changed.
     * @param attributesAffected
     *            An arbitrary number of attributes that have been changed.
     */
    public static void logConfigChange(
        @Nonnull final AccessRequester initiator,
        @Nonnull final AuditedDataObject object,
        @Nullable final Throwable error,
        @Nonnull final AccessedAttribute attributeAffected,
        @Nullable final AccessedAttribute... attributesAffected )
    {
        getAuditLog().logConfigChange(initiator, object, error, attributeAffected, attributesAffected);
    }

    /**
     * Logs the change of some attributes of configuration data by the current request.
     *
     * @param object
     *            An instance of {@link AuditedDataObject} that represents the object that has been modified.
     * @param error
     *            The exception in case of an error.
     * @param attributeAffected
     *            An attribute that has been changed.
     * @param attributesAffected
     *            An arbitrary number of attributes that have been changed.
     */
    public static void logConfigChange(
        @Nonnull final AuditedDataObject object,
        @Nullable final Throwable error,
        @Nonnull final AccessedAttribute attributeAffected,
        @Nullable final AccessedAttribute... attributesAffected )
    {
        logConfigChange(AccessRequester.ofCurrentRequest(), object, error, attributeAffected, attributesAffected);
    }

    /**
     * Logs the <b>attempt</b> to read some attributes from an object.
     *
     * @param initiator
     *            Information about the user trying to the read the attributes.
     * @param object
     *            An instance of {@link AuditedDataObject} that represents the object to be read.
     * @param subject
     *            An instance of {@link AuditedDataSubject} that represents the owner of the data to be read.
     * @param attributeAffected
     *            An attribute that should be read.
     * @param attributesAffected
     *            An arbitrary number of attributes that should be read.
     */
    public static void logDataReadAttempt(
        @Nonnull final AccessRequester initiator,
        @Nonnull final AuditedDataObject object,
        @Nonnull final AuditedDataSubject subject,
        @Nonnull final AccessedAttribute attributeAffected,
        @Nullable final AccessedAttribute... attributesAffected )
    {
        getAuditLog().logDataReadAttempt(initiator, object, subject, attributeAffected, attributesAffected);
    }

    /**
     * Logs the <b>attempt</b> to read some attributes from an object by the current request.
     *
     * @param object
     *            An instance of {@link AuditedDataObject} that represents the object to be read.
     * @param subject
     *            An instance of {@link AuditedDataSubject} that represents the owner of the data to be read.
     * @param attributeAffected
     *            An attribute that should be read.
     * @param attributesAffected
     *            An arbitrary number of attributes that should be read.
     */
    public static void logDataReadAttempt(
        @Nonnull final AuditedDataObject object,
        @Nonnull final AuditedDataSubject subject,
        @Nonnull final AccessedAttribute attributeAffected,
        @Nullable final AccessedAttribute... attributesAffected )
    {
        logDataReadAttempt(AccessRequester.ofCurrentRequest(), object, subject, attributeAffected, attributesAffected);
    }

    /**
     * Logs the read access to some attributes of an object.
     *
     * @param initiator
     *            Information about the user reading the attributes.
     * @param object
     *            An instance of {@link AuditedDataObject} that represents the object that has been read.
     * @param subject
     *            An instance of {@link AuditedDataSubject} that represents the owner of the data that has been read.
     * @param error
     *            The exception in case of an error.
     * @param attributeAffected
     *            An attribute that has been read.
     * @param attributesAffected
     *            An arbitrary number of attributes that have been read.
     */
    public static void logDataRead(
        @Nonnull final AccessRequester initiator,
        @Nonnull final AuditedDataObject object,
        @Nonnull final AuditedDataSubject subject,
        @Nullable final Throwable error,
        @Nonnull final AccessedAttribute attributeAffected,
        @Nullable final AccessedAttribute... attributesAffected )
    {
        getAuditLog().logDataRead(initiator, object, subject, error, attributeAffected, attributesAffected);
    }

    /**
     * Logs the read access to some attributes of an object by the current request.
     *
     * @param object
     *            An instance of {@link AuditedDataObject} that represents the object that has been read.
     * @param subject
     *            An instance of {@link AuditedDataSubject} that represents the owner of the data that has been read.
     * @param error
     *            The exception in case of an error.
     * @param attributeAffected
     *            An attribute that has been read.
     * @param attributesAffected
     *            An arbitrary number of attributes that have been read.
     */
    public static void logDataRead(
        @Nonnull final AuditedDataObject object,
        @Nonnull final AuditedDataSubject subject,
        @Nullable final Throwable error,
        @Nonnull final AccessedAttribute attributeAffected,
        @Nullable final AccessedAttribute... attributesAffected )
    {
        logDataRead(AccessRequester.ofCurrentRequest(), object, subject, error, attributeAffected, attributesAffected);
    }

    /**
     * Logs the attempt to write to some attributes of an object.
     *
     * @param initiator
     *            Information about the user trying to write to some attributes.
     * @param object
     *            An instance of {@link AuditedDataObject} that represents the object that is about to be modified.
     * @param subject
     *            An instance of {@link AuditedDataSubject} that represents the owner of the data that is about to be
     *            modified.
     * @param attributeAffected
     *            An attribute that should be modified, containing the old and new value.
     * @param attributesAffected
     *            An arbitrary number of attributes that should be modified, containing the old and new values.
     */
    public static void logDataWriteAttempt(
        @Nonnull final AccessRequester initiator,
        @Nonnull final AuditedDataObject object,
        @Nonnull final AuditedDataSubject subject,
        @Nonnull final AccessedAttribute attributeAffected,
        @Nullable final AccessedAttribute... attributesAffected )
    {
        getAuditLog().logDataWriteAttempt(initiator, object, subject, attributeAffected, attributesAffected);
    }

    /**
     * Logs the attempt to write to some attributes of an object by the current request.
     *
     * @param object
     *            An instance of {@link AuditedDataObject} that represents the object that is about to be modified.
     * @param subject
     *            An instance of {@link AuditedDataSubject} that represents the owner of the data that is about to be
     *            modified.
     * @param attributeAffected
     *            An attribute that should be modified, containing the old and new value.
     * @param attributesAffected
     *            An arbitrary number of attributes that should be modified, containing the old and new values.
     */
    public static void logDataWriteAttempt(
        @Nonnull final AuditedDataObject object,
        @Nonnull final AuditedDataSubject subject,
        @Nonnull final AccessedAttribute attributeAffected,
        @Nullable final AccessedAttribute... attributesAffected )
    {
        logDataWriteAttempt(AccessRequester.ofCurrentRequest(), object, subject, attributeAffected, attributesAffected);
    }

    /**
     * Logs the modification of some attributes of an object.
     *
     * @param initiator
     *            Information about the user changing the attributes.
     * @param object
     *            An instance of {@link AuditedDataObject} that represents the object that has been modified.
     * @param subject
     *            An instance of {@link AuditedDataSubject} that represents the owner of the data that has been
     *            modified.
     * @param error
     *            The exception in case of an error.
     * @param attributeAffected
     *            An attribute that has been modified, containing the old and new value.
     * @param attributesAffected
     *            An arbitrary number of attributes that have been modified, containing the old and new values.
     */
    public static void logDataWrite(
        @Nonnull final AccessRequester initiator,
        @Nonnull final AuditedDataObject object,
        @Nonnull final AuditedDataSubject subject,
        @Nullable final Throwable error,
        @Nonnull final AccessedAttribute attributeAffected,
        @Nullable final AccessedAttribute... attributesAffected )
    {
        getAuditLog().logDataWrite(initiator, object, subject, error, attributeAffected, attributesAffected);
    }

    /**
     * Logs the modification of some attributes of an object by the current request.
     *
     * @param object
     *            An instance of {@link AuditedDataObject} that represents the object that has been modified.
     * @param subject
     *            An instance of {@link AuditedDataSubject} that represents the owner of the data that has been
     *            modified.
     * @param error
     *            The exception in case of an error.
     * @param attributeAffected
     *            An attribute that has been modified, containing the old and new value.
     * @param attributesAffected
     *            An arbitrary number of attributes that have been modified, containing the old and new values.
     */
    public static void logDataWrite(
        @Nonnull final AuditedDataObject object,
        @Nonnull final AuditedDataSubject subject,
        @Nullable final Throwable error,
        @Nonnull final AccessedAttribute attributeAffected,
        @Nullable final AccessedAttribute... attributesAffected )
    {
        logDataWrite(AccessRequester.ofCurrentRequest(), object, subject, error, attributeAffected, attributesAffected);
    }
}
