/*
 * Copyright (c) 2022 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.auditlog;

import static com.sap.cloud.sdk.cloudplatform.auditlog.AuditLogUtils.attributesAsList;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Joiner;

/**
 * Implementation of {@link AuditLog} that redirects to the default logger.
 * <p>
 * <strong>This class does not guarantee any qualities that are expected for productive auditing purposes!</strong>
 */
public class DefaultLoggerAuditLog implements AuditLog
{
    private static final String ENTRY_SEPARATOR = " | ";
    private static final String UNKNOWN_VALUE = "unknown";

    private final Logger logger;

    /**
     * Default constructor. Uses a logger from the {@link LoggerFactory} as logger implementation.
     */
    public DefaultLoggerAuditLog()
    {
        logger = LoggerFactory.getLogger(DefaultLoggerAuditLog.class);
    }

    DefaultLoggerAuditLog( @Nonnull final Logger logger )
    {
        this.logger = logger;
    }

    private void logToLocal(
        final boolean isBeginning,
        @Nonnull final String topic,
        @Nonnull final AccessRequester initiator,
        @Nullable final String message,
        @Nullable final Throwable e )
    {
        final String initiatorMessage =
            "INITIATOR: ["
                + Joiner.on(", ").join(
                    "User ID: " + initiator.getPrincipalId().getOrElse(UNKNOWN_VALUE),
                    "Tenant ID: " + initiator.getTenantId().getOrElse(UNKNOWN_VALUE),
                    "IP Address: " + initiator.getIpAddress().getOrElse(UNKNOWN_VALUE),
                    "Channel: " + initiator.getChannel().getOrElse(UNKNOWN_VALUE))
                + "]";

        final Joiner messageJoiner = Joiner.on(ENTRY_SEPARATOR).skipNulls();
        if( e == null ) {
            final String phase = isBeginning ? "[BEGINNING]" : "[COMPLETED]";
            logger.info(messageJoiner.join(topic, phase, initiatorMessage, message));
        } else {
            logger.error(messageJoiner.join(topic, "[FAILED]", initiatorMessage, message), e);
        }
    }

    @Nonnull
    private String auditObjectToLogMessage( @Nonnull final AuditedDataObject object )
    {
        return "OBJECT: ["
            + Joiner.on(", ").join("Type: " + object.getType(), "Identifiers: " + object.getAllIdentifiers())
            + "]";
    }

    @Nonnull
    private String auditSubjectToLogMessage( @Nonnull final AuditedDataSubject subject )
    {
        return "SUBJECT: ["
            + Joiner.on(", ").join(
                "Type: " + subject.getType(),
                "Role: " + subject.getRole(),
                "Identifiers: " + subject.getAllIdentifiers())
            + "]";
    }

    @Nonnull
    private String attributesToLogMessage( @Nullable final Iterable<AccessedAttribute> attributes )
    {
        final StringBuilder message = new StringBuilder("ATTRIBUTES: [");
        if( attributes != null ) {
            final ObjectMapper jsonMapper = new ObjectMapper();

            for( final AccessedAttribute attribute : attributes ) {
                try {
                    message.append(jsonMapper.writeValueAsString(attribute));
                }
                catch( final JsonProcessingException e ) {
                    message.append("{").append(attribute).append("}");
                }
            }
        }
        message.append("]");
        return message.toString();
    }

    private void logSecurityEvent(
        final boolean isBeginning,
        @Nonnull final AccessRequester initiator,
        @Nullable final String message,
        @Nullable final Throwable error )
    {
        logToLocal(isBeginning, "[SECURITY]", initiator, message, error);
    }

    private void logConfigChange(
        final boolean isBeginning,
        @Nonnull final AccessRequester initiator,
        @Nonnull final AuditedDataObject object,
        @Nullable final Iterable<AccessedAttribute> attributesAffected,
        @Nullable final Throwable error )
    {
        final String message =
            Joiner
                .on(ENTRY_SEPARATOR)
                .join(auditObjectToLogMessage(object), attributesToLogMessage(attributesAffected));
        logToLocal(isBeginning, "[CONFIG]", initiator, message, error);
    }

    private void logDataRead(
        final boolean isBeginning,
        @Nonnull final AccessRequester initiator,
        @Nonnull final AuditedDataObject object,
        @Nonnull final AuditedDataSubject subject,
        @Nullable final Iterable<AccessedAttribute> attributesAffected,
        @Nullable final Throwable error )
    {
        final String message =
            Joiner.on(ENTRY_SEPARATOR).join(
                auditObjectToLogMessage(object),
                auditSubjectToLogMessage(subject),
                attributesToLogMessage(attributesAffected));
        logToLocal(isBeginning, "[DATA-READ]", initiator, message, error);
    }

    private void logDataWrite(
        final boolean isBeginning,
        @Nonnull final AccessRequester initiator,
        @Nonnull final AuditedDataObject object,
        @Nonnull final AuditedDataSubject subject,
        @Nullable final Iterable<AccessedAttribute> attributesAffected,
        @Nullable final Throwable error )
    {
        final String message =
            Joiner.on(ENTRY_SEPARATOR).join(
                auditObjectToLogMessage(object),
                auditSubjectToLogMessage(subject),
                attributesToLogMessage(attributesAffected));
        logToLocal(isBeginning, "[DATA-WRITE]", initiator, message, error);
    }

    @Override
    public void logSecurityEventBeginning( @Nonnull final AccessRequester initiator, @Nullable final String message )
    {
        logSecurityEvent(true, initiator, message, null);
    }

    @Override
    public void logSecurityEvent(
        @Nonnull final AccessRequester initiator,
        @Nullable final String message,
        @Nullable final Throwable throwable )
    {
        logSecurityEvent(false, initiator, message, throwable);
    }

    @Override
    public void logConfigChangeBeginning(
        @Nonnull final AccessRequester initiator,
        @Nonnull final AuditedDataObject object,
        @Nonnull final AccessedAttribute attributeAffected,
        @Nullable final AccessedAttribute... attributesAffected )
    {
        logConfigChange(true, initiator, object, attributesAsList(attributeAffected, attributesAffected), null);
    }

    @Override
    public void logConfigChange(
        @Nonnull final AccessRequester initiator,
        @Nonnull final AuditedDataObject object,
        @Nullable final Throwable error,
        @Nonnull final AccessedAttribute attributeAffected,
        @Nullable final AccessedAttribute... attributesAffected )
    {
        logConfigChange(false, initiator, object, attributesAsList(attributeAffected, attributesAffected), error);
    }

    @Override
    public void logDataReadAttempt(
        @Nonnull final AccessRequester initiator,
        @Nonnull final AuditedDataObject object,
        @Nonnull final AuditedDataSubject subject,
        @Nonnull final AccessedAttribute attributeAffected,
        @Nullable final AccessedAttribute... attributesAffected )
    {
        logDataRead(true, initiator, object, subject, attributesAsList(attributeAffected, attributesAffected), null);
    }

    @Override
    public void logDataRead(
        @Nonnull final AccessRequester initiator,
        @Nonnull final AuditedDataObject object,
        @Nonnull final AuditedDataSubject subject,
        @Nullable final Throwable error,
        @Nonnull final AccessedAttribute attributeAffected,
        @Nullable final AccessedAttribute... attributesAffected )
    {
        logDataRead(false, initiator, object, subject, attributesAsList(attributeAffected, attributesAffected), error);
    }

    @Override
    public void logDataWriteAttempt(
        @Nonnull final AccessRequester initiator,
        @Nonnull final AuditedDataObject object,
        @Nonnull final AuditedDataSubject subject,
        @Nonnull final AccessedAttribute attributeAffected,
        @Nullable final AccessedAttribute... attributesAffected )
    {
        logDataWrite(true, initiator, object, subject, attributesAsList(attributeAffected, attributesAffected), null);
    }

    @Override
    public void logDataWrite(
        @Nonnull final AccessRequester initiator,
        @Nonnull final AuditedDataObject object,
        @Nonnull final AuditedDataSubject subject,
        @Nullable final Throwable error,
        @Nonnull final AccessedAttribute attributeAffected,
        @Nullable final AccessedAttribute... attributesAffected )
    {
        logDataWrite(false, initiator, object, subject, attributesAsList(attributeAffected, attributesAffected), error);
    }
}
