/*
 * Copyright (c) 2022 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.auditlog;

import java.util.Map;
import java.util.UUID;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.common.collect.Maps;

import lombok.Getter;
import lombok.Setter;

/**
 * This class stores auditing information about the object being accessed. The type and keys should be set such that
 * from the audit log entry, one can find the object in the application's persistence (e.g. database).
 */
public class AuditedDataObject
{
    /**
     * This represents the type of the object being accessed.
     */
    @Getter
    @Setter
    private String type;

    private final Map<String, String> identifiers;

    /**
     * Creates an audited data object instance with provided object type and randomly generated identifier.
     *
     * @param type
     *            The type of the object being accessed.
     */
    public AuditedDataObject( @Nonnull final String type )
    {
        this(type, UUID.randomUUID().toString());
    }

    /**
     * Creates an audited data object instance with provided object type and identifier.
     *
     * @param type
     *            The type of the object being accessed.
     * @param id
     *            The identifier value for property "AuditedDataObject".
     */
    public AuditedDataObject( @Nonnull final String type, @Nonnull final String id )
    {
        this.type = type;
        identifiers = Maps.newHashMap();
        identifiers.put(AuditedDataObject.class.getSimpleName(), id);
    }

    /**
     * Sets a key property that identifies the accessed object. Entries can be overwritten if the same
     * {@code propertyName} is passed in.
     *
     * @param propertyName
     *            Name of the key property.
     * @param value
     *            Value of the key property.
     */
    public void setIdentifier( @Nonnull final String propertyName, @Nullable final String value )
    {
        identifiers.put(propertyName, value);
    }

    /**
     * Gets the value of a key property that identifies the accessed object.
     *
     * @param propertyName
     *            Name of the key property.
     * @return Value of the key property.
     */
    @Nullable
    public String getIdentifier( @Nonnull final String propertyName )
    {
        return identifiers.get(propertyName);
    }

    /**
     * Gets all properties that identify the accessed object.
     *
     * @return Map of name-value pairs.
     */
    @Nonnull
    public Map<String, String> getAllIdentifiers()
    {
        return identifiers;
    }
}
