/*
 * Copyright (c) 2023 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.auditlog;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

/**
 * Utility class providing functions common to all AuditLog implementations.
 */
public class AuditLogUtils
{
    /**
     * Takes an instance of {@link AccessedAttribute} as well as an arbitrary number of additional instances and merges
     * them into a list.
     * 
     * @param attribute
     *            An instance of {@link AccessedAttribute}.
     * @param attributes
     *            An arbitrary number of additional instances of {@link AccessedAttribute}.
     * @return A merged list of both parameters (with the single one at index 0).
     */
    @Nonnull
    public static
        List<AccessedAttribute>
        attributesAsList( @Nonnull final AccessedAttribute attribute, @Nullable final AccessedAttribute... attributes )
    {
        final List<AccessedAttribute> attributesAsList =
            attributes == null || attributes.length == 0 ? new ArrayList<>() : list(attributes);
        attributesAsList.add(0, attribute);
        return attributesAsList;
    }

    /**
     * Arrays.asList() returns a list that does not support add(int, T)...
     *
     * @return A list.
     */
    private static List<AccessedAttribute> list( final AccessedAttribute[] accessedAttributes )
    {
        return Arrays.stream(accessedAttributes).collect(Collectors.toList());
    }
}
