/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.auditlog;

import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.sap.cloud.sdk.cloudplatform.security.principal.Principal;
import com.sap.cloud.sdk.cloudplatform.security.principal.PrincipalAccessor;
import com.sap.cloud.sdk.cloudplatform.tenant.Tenant;
import com.sap.cloud.sdk.cloudplatform.tenant.TenantAccessor;
import com.sap.cloud.sdk.cloudplatform.thread.ThreadContext;
import com.sap.cloud.sdk.cloudplatform.thread.ThreadContextAccessor;

import io.vavr.control.Option;
import lombok.AllArgsConstructor;
import lombok.Data;

/**
 * Represents the requester of an access to an object in the context of an audit log.
 * <p>
 * Therefore this class stores identifiable information of the requester. Information about the field accessed as well
 * as about the access time are handled in the implementation of the {@link AuditLog} interface.
 */
@Data
@AllArgsConstructor
public class AccessRequester
{
    /**
     * Field representing an unknown requester.
     */
    public static final AccessRequester UNKNOWN = new AccessRequester(null, null, null, null);

    /**
     * Default user Id, used in case of an unknown remote user.
     */
    public static final String UNAUTHENTICATED_USER = "noauth";

    /**
     * Tenant Id of the user requesting the access.
     */
    @Nullable
    private final String tenantId;

    /**
     * Id of the user requesting the access.
     */
    @Nullable
    private final String principalId;

    /**
     * The IP address of the user requesting the access.
     */
    @Nullable
    private final String ipAddress;

    /**
     * The connection channel used, e.g. http, or https.
     */
    @Nullable
    private final String channel;

    /**
     * The Id of the remote user requesting access.
     *
     * @return An {@link Option} containing the user Id.
     */
    @Nonnull
    public Option<String> getPrincipalId()
    {
        return Option.of(principalId);
    }

    /**
     * The tenant or zone Id of the remote user requesting access.
     *
     * @return An {@link Option} containing the tenant or zone Id.
     */
    @Nonnull
    public Option<String> getTenantId()
    {
        return Option.of(tenantId);
    }

    /**
     * The IP address of the remote user requesting access.
     *
     * @return An {@link Option} containing the IP address.
     */
    @Nonnull
    public Option<String> getIpAddress()
    {
        return Option.of(ipAddress);
    }

    /**
     * The connection channel used, e.g. http, or https.
     *
     * @return An {@link Option} containing the connection channel.
     */
    @Nonnull
    public Option<String> getChannel()
    {
        return Option.of(channel);
    }

    /**
     * Creates an AccessRequester based on the given request.
     *
     * @param request
     *            The request which tries to access an object.
     */
    public AccessRequester( @Nonnull final javax.servlet.http.HttpServletRequest request )
    {
        this(null, getCurrentPrincipalId(), request.getRemoteAddr(), request.getScheme());
    }

    /**
     * Creates an AccessRequester based on the given request and tenant.
     *
     * @param request
     *            The request which tries to access an object.
     * @param tenant
     *            The tenant which tries to access an object.
     */
    public AccessRequester( @Nonnull final javax.servlet.http.HttpServletRequest request, @Nonnull final Tenant tenant )
    {
        this(tenant.getTenantId(), getCurrentPrincipalId(), request.getRemoteAddr(), request.getScheme());
    }

    /**
     * Creates a request based on the given, nullable request.
     *
     * @param request
     *            The request which tries to access an object.
     *
     * @return An AccessRequester for the given request. Never {@code null}.
     */
    @Nonnull
    public static AccessRequester of( @Nullable final javax.servlet.http.HttpServletRequest request )
    {
        if( request == null ) {
            return AccessRequester.UNKNOWN;
        }
        return new AccessRequester(request);
    }

    /**
     * Creates a request based on the given, nullable request and tenant.
     *
     * @param request
     *            The request which tries to access an object.
     * @param tenant
     *            The tenant which tries to access an object.
     *
     * @return An AccessRequester for the given request. Never {@code null}.
     */
    @Nonnull
    public static
        AccessRequester
        of( @Nullable final javax.servlet.http.HttpServletRequest request, @Nullable final Tenant tenant )
    {
        if( request != null ) {
            if( tenant != null ) {
                return new AccessRequester(request, tenant);
            } else {
                return new AccessRequester(request);
            }
        } else {
            return AccessRequester.UNKNOWN;
        }
    }

    /**
     * Creates a request based on the given, nullable request.
     *
     * @param request
     *            The request which tries to access an object.
     *
     * @return An AccessRequester for the given request. Never {@code null}.
     */
    @Nonnull
    public static AccessRequester ofJakarta( @Nullable final jakarta.servlet.http.HttpServletRequest request )
    {
        return ofJakarta(request, null);
    }

    /**
     * Creates a request based on the given, nullable request.
     *
     * @param req
     *            The request which tries to access an object.
     * @param tenant
     *            The tenant which tries to access an object.
     *
     * @return An AccessRequester for the given request. Never {@code null}.
     */
    @Nonnull
    public static
        AccessRequester
        ofJakarta( @Nullable final jakarta.servlet.http.HttpServletRequest req, @Nullable final Tenant tenant )
    {
        if( req == null ) {
            return AccessRequester.UNKNOWN;
        }
        if( tenant == null ) {
            return new AccessRequester(null, getCurrentPrincipalId(), req.getRemoteAddr(), req.getScheme());
        }
        return new AccessRequester(tenant.getTenantId(), getCurrentPrincipalId(), req.getRemoteAddr(), req.getScheme());
    }

    /**
     * Creates a request based on the current request.
     *
     * @return An AccessRequester for the current request. Never {@code null}.
     */
    @Nonnull
    public static AccessRequester ofCurrentRequest()
    {
        final String PROPERTY_SERVLET_SCHEME = "servlet-scheme";
        final String PROPERTY_SERVLET_REMOTE_ADDRESS = "servlet-remote-address";

        @Nullable
        final String tenantId = TenantAccessor.tryGetCurrentTenant().map(Tenant::getTenantId).getOrNull();

        final ThreadContext tc = ThreadContextAccessor.getCurrentContextOrNull();
        final Function<String, String> threadContextExtractor =
            ( prop ) -> tc != null && tc.containsProperty(prop) ? tc.<String> getPropertyValue(prop).getOrNull() : null;

        final String scheme = threadContextExtractor.apply(PROPERTY_SERVLET_SCHEME);
        final String remoteAddress = threadContextExtractor.apply(PROPERTY_SERVLET_REMOTE_ADDRESS);
        return new AccessRequester(tenantId, getCurrentPrincipalId(), remoteAddress, scheme);
    }

    @Nonnull
    private static String getCurrentPrincipalId()
    {
        return PrincipalAccessor
            .tryGetCurrentPrincipal()
            .map(Principal::getPrincipalId)
            .getOrElse(UNAUTHENTICATED_USER);
    }
}
