/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.auditlog;

import java.util.Map;
import java.util.UUID;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.common.collect.Maps;

import lombok.Getter;
import lombok.Setter;

/**
 * This class stores auditing information about who owns the data. Fields should be set such that from the audit log
 * entry, one can identify the owner of the accessed data.
 */
public class AuditedDataSubject
{

    /**
     * This represents the type of owner for the data (e.g. person, company, ...).
     */
    @Getter
    @Setter
    private String type;

    /**
     * This represents the role of the owner for the data (e.g. administrator, controller, ...).
     */
    @Getter
    @Setter
    private String role;

    private final Map<String, String> identifiers;

    /**
     * Creates an audited data subject instance with provided type and role and randomly generated identifier.
     *
     * @param type
     *            The type of owner for the data (e.g. person, company, ...).
     * @param role
     *            The role of the owner for the data (e.g. administrator, controller, ...).
     */
    public AuditedDataSubject( @Nonnull final String type, @Nonnull final String role )
    {
        this(type, role, UUID.randomUUID().toString());
    }

    /**
     * Creates an audited data subject instance with provided type, role and identifier.
     *
     * @param type
     *            The type of owner for the data (e.g. person, company, ...).
     * @param role
     *            The role of the owner for the data (e.g. administrator, controller, ...).
     * @param id
     *            The identifier value for property "AuditedDataObject".
     */
    public AuditedDataSubject( @Nonnull final String type, @Nonnull final String role, @Nonnull final String id )
    {
        this.type = type;
        this.role = role;
        identifiers = Maps.newHashMap();
        identifiers.put(AuditedDataSubject.class.getSimpleName(), id);
    }

    /**
     * Sets a property that identifies the data owner. Entries can be overwritten if the same {@code propertyName} is
     * passed in.
     *
     * @param propertyName
     *            Name of the property.
     * @param value
     *            Value of the property.
     */
    public void setIdentifier( @Nonnull final String propertyName, @Nullable final String value )
    {
        identifiers.put(propertyName, value);
    }

    /**
     * Gets the value of a property that identifies the data owner.
     *
     * @param propertyName
     *            Name of the property.
     * @return Value of the property.
     */
    @Nullable
    public String getIdentifier( @Nonnull final String propertyName )
    {
        return identifiers.get(propertyName);
    }

    /**
     * Gets all properties that identify the data owner.
     *
     * @return Map of name-value pairs.
     */
    @Nonnull
    public Map<String, String> getAllIdentifiers()
    {
        return identifiers;
    }
}
