/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.connectivity;

import java.security.KeyStore;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.HttpClient;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;

import com.sap.cloud.security.client.HttpClientFactory;
import com.sap.cloud.security.config.ClientIdentity;
import com.sap.cloud.security.mtls.SSLContextFactory;

import io.vavr.control.Try;
import lombok.extern.slf4j.Slf4j;

/**
 * Creates a {@link CloseableHttpClient} instance. Supports certificate based communication.
 *
 * @deprecated The class is no longer required and will be removed with the next major version.
 */
@Deprecated
@Slf4j
public class CertificateBasedHttpClientFactory implements HttpClientFactory
{
    @Override
    @Nonnull
    public CloseableHttpClient createClient( @Nullable final ClientIdentity clientIdentity )
    {
        log.info("Using CertificateBasedHttpClientFactory service");
        if( clientIdentity == null || !clientIdentity.isCertificateBased() ) {
            log.debug("Setting up default http client");
            return HttpClients.createDefault();
        }

        log
            .debug(
                "Trying to setup a HTTPS client with certificate: {} and private key: {}",
                clientIdentity.getCertificate(),
                clientIdentity.getKey());

        final KeyStore keyStore = loadKeyStore(clientIdentity);

        // we must use the "hacky" approach of using an empty URI here because we do not have a URI at hand
        final HttpDestinationProperties httpDestinationProperties =
            DefaultHttpDestination
                .builder("")
                .securityConfiguration(SecurityConfigurationStrategy.FROM_DESTINATION)
                .authenticationType(AuthenticationType.NO_AUTHENTICATION)
                .keyStore(keyStore)
                .build();

        final HttpClient httpClient =
            HttpClientAccessor.getHttpClientFactory().createHttpClient(httpDestinationProperties);

        if( httpClient instanceof CloseableHttpClient ) {
            return (CloseableHttpClient) httpClient;
        }

        throw new IllegalStateException(
            String
                .format(
                    "Unable to instantiate a new HTTPS client using certificate based authentication. The created client is an instance of \"%s\" but we expected it to be a \"%s\".",
                    httpClient.getClass(),
                    CloseableHttpClient.class));

    }

    @Nonnull
    private static KeyStore loadKeyStore( @Nonnull final ClientIdentity clientIdentity )
    {
        return Try
            .of(() -> SSLContextFactory.getInstance().createKeyStore(clientIdentity))
            .getOrElseThrow(
                e -> new IllegalArgumentException(
                    "Unable to create a key store from the provided client identity.",
                    e));
    }
}
