/*
 * Copyright (c) 2019 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.resilience;

import java.util.concurrent.Callable;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.function.Function;
import java.util.function.Supplier;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import io.vavr.control.Try;

/**
 * Generic interface to decorate functions with non-functional requirements.
 */
public interface ResilienceDecorationStrategy
{
    /**
     * Clears the cache associated with the given {@link ResilienceConfiguration} if any exists. Be aware that the cache
     * will be cleared for all tenants and principals.
     *
     * @param configuration
     *            The {@code ResilienceConfiguration} the cache is attached to.
     */
    default void invalidateCache( @Nonnull final ResilienceConfiguration configuration )
    {
    }

    /**
     * Decorate an instance of a supplier function.
     *
     * @param supplier
     *            The supplier.
     * @param configuration
     *            The configuration of the resilient call.
     * @param <T>
     *            The return type of the call.
     *
     * @return A decorated supplier.
     */
    @Nonnull
    default <T> Supplier<T> decorateSupplier(
        @Nonnull final Supplier<T> supplier,
        @Nonnull final ResilienceConfiguration configuration )
    {
        return decorateSupplier(supplier, configuration, null);
    }

    /**
     * Decorate and execute an instance of a supplier function.
     *
     * @param supplier
     *            The supplier.
     * @param configuration
     *            The configuration of the resilient call.
     * @param <T>
     *            The return type of the call.
     *
     * @return The value of the supplier.
     */
    @Nullable
    default <T> T executeSupplier(
        @Nonnull final Supplier<T> supplier,
        @Nonnull final ResilienceConfiguration configuration )
    {
        return decorateSupplier(supplier, configuration).get();
    }

    /**
     * Decorate an instance of a supplier function.
     *
     * @param supplier
     *            The supplier.
     * @param configuration
     *            The configuration of the resilient call.
     * @param fallbackFunction
     *            In case of failure, execute this function.
     * @param <T>
     *            The return type of the call.
     *
     * @return A decorated supplier.
     */
    @Nonnull
    <T> Supplier<T> decorateSupplier(
        @Nonnull final Supplier<T> supplier,
        @Nonnull final ResilienceConfiguration configuration,
        @Nullable final Function<? super Throwable, T> fallbackFunction );

    /**
     * Decorate and execute an instance of a supplier function.
     *
     * @param supplier
     *            The supplier.
     * @param configuration
     *            The configuration of the resilient call.
     * @param fallbackFunction
     *            In case of failure, execute this function.
     * @param <T>
     *            The return type of the call.
     *
     * @return The value of the supplier.
     */
    @Nullable
    default <T> T executeSupplier(
        @Nonnull final Supplier<T> supplier,
        @Nonnull final ResilienceConfiguration configuration,
        @Nullable final Function<? super Throwable, T> fallbackFunction )
    {
        return decorateSupplier(supplier, configuration, fallbackFunction).get();
    }

    /**
     * Decorate an instance of a callable function.
     *
     * @param callable
     *            The callable.
     * @param configuration
     *            The configuration of the resilient call.
     * @param <T>
     *            The return type of the call.
     *
     * @return A decorated callable.
     */
    @Nonnull
    default <T> Callable<T> decorateCallable(
        @Nonnull final Callable<T> callable,
        @Nonnull final ResilienceConfiguration configuration )
    {
        return decorateCallable(callable, configuration, null);
    }

    /**
     * Decorate and execute an instance of a callable function.
     *
     * @param callable
     *            The callable.
     * @param configuration
     *            The configuration of the resilient call.
     * @param <T>
     *            The return type of the call.
     *
     * @throws Exception
     *             Exception that can be thrown by the callable.
     *
     * @return The value returned by the callable.
     */
    @SuppressWarnings( "PMD.SignatureDeclareThrowsException" )
    @Nullable
    default <T> T executeCallable(
        @Nonnull final Callable<T> callable,
        @Nonnull final ResilienceConfiguration configuration )
        throws Exception
    {
        return decorateCallable(callable, configuration).call();
    }

    /**
     * Decorate an instance of a callable function.
     *
     * @param callable
     *            The callable.
     * @param configuration
     *            The configuration of the resilient call.
     * @param fallbackFunction
     *            In case of failure, execute this function.
     * @param <T>
     *            The return type of the call.
     *
     * @return A decorated callable.
     */
    @Nonnull
    <T> Callable<T> decorateCallable(
        @Nonnull final Callable<T> callable,
        @Nonnull final ResilienceConfiguration configuration,
        @Nullable final Function<? super Throwable, T> fallbackFunction );

    /**
     * Decorate and execute an instance of a callable function. This method assumes that {@code fallbackFunction}
     * handles any exceptions thrown by {@code callable}, and {@code fallbackFunction} does not throw any exceptions
     * itself. If {@code fallbackFunction} throws any exception, then it will be wrapped in a
     * {@link ResilienceRuntimeException}.
     *
     * @param callable
     *            The callable.
     * @param configuration
     *            The configuration of the resilient call.
     * @param <T>
     *            The return type of the call.
     *
     * @return The value returned by the callable.
     */
    @Nullable
    default <T> T executeCallable(
        @Nonnull final Callable<T> callable,
        @Nonnull final ResilienceConfiguration configuration,
        @Nullable final Function<? super Throwable, T> fallbackFunction )
    {
        try {
            return decorateCallable(callable, configuration, fallbackFunction).call();
        }
        // CHECKSTYLE:OFF
        catch( final Throwable e ) {
            throw new ResilienceRuntimeException("The provided fallback method threw an exception. ", e);
        }
        // CHECKSTYLE:ON
    }

    /**
     * Decorate an instance of a callable function. Automatically schedule the asynchronous execution.
     *
     * @param callable
     *            The callable.
     * @param configuration
     *            The configuration of the resilient call.
     * @param fallbackFunction
     *            (Optional) In case of failure, execute this function.
     * @param executor
     *            (Optional) The thread executor to schedule the asynchronous operation with.
     * @param <T>
     *            The return type of the call.
     *
     * @return An instance of Future being executed asynchronously with the provided (or default) executor.
     */
    @Nonnull
    default <T> CompletableFuture<T> queueCallable(
        @Nonnull final Callable<T> callable,
        @Nonnull final ResilienceConfiguration configuration,
        @Nullable final Function<? super Throwable, T> fallbackFunction,
        @Nullable final Executor executor )
    {
        final Callable<T> call = decorateCallable(callable, configuration, fallbackFunction);
        final Supplier<T> supp = Try.ofCallable(call)::get;
        return executor != null ? CompletableFuture.supplyAsync(supp, executor) : CompletableFuture.supplyAsync(supp);
    }

    /**
     * Decorate an instance of a supplier function. Automatically schedule the asynchronous execution.
     *
     * @param supplier
     *            The supplier.
     * @param configuration
     *            The configuration of the resilient call.
     * @param fallbackFunction
     *            (Optional) In case of failure, execute this function.
     * @param executor
     *            (Optional) The thread executor to schedule the asynchronous operation with.
     * @param <T>
     *            The return type of the call.
     *
     * @return An instance of Future being executed asynchronously with the provided (or default) executor.
     */
    @Nonnull
    default <T> CompletableFuture<T> queueSupplier(
        @Nonnull final Supplier<T> supplier,
        @Nonnull final ResilienceConfiguration configuration,
        @Nullable final Function<? super Throwable, T> fallbackFunction,
        @Nullable final Executor executor )
    {
        final Supplier<T> func = decorateSupplier(supplier, configuration, fallbackFunction);
        return executor != null ? CompletableFuture.supplyAsync(func, executor) : CompletableFuture.supplyAsync(func);
    }
}
