/*
 * Copyright (c) 2022 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security;

import java.util.concurrent.Callable;
import java.util.concurrent.Future;
import java.util.function.Supplier;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.auth0.jwt.interfaces.DecodedJWT;
import com.sap.cloud.sdk.cloudplatform.security.exception.AuthTokenAccessException;
import com.sap.cloud.sdk.cloudplatform.security.exception.TokenRequestDeniedException;
import com.sap.cloud.sdk.cloudplatform.security.exception.TokenRequestFailedException;
import com.sap.cloud.sdk.cloudplatform.thread.DefaultThreadContext;
import com.sap.cloud.sdk.cloudplatform.thread.Executable;
import com.sap.cloud.sdk.cloudplatform.thread.ThreadContextExecutor;
import com.sap.cloud.sdk.cloudplatform.thread.exception.ThreadContextExecutionException;

import io.vavr.control.Try;
import lombok.AccessLevel;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;

/**
 * Accessor for retrieving the current {@link AuthToken} on SAP Cloud Platform Cloud Foundry.
 */
@NoArgsConstructor( access = AccessLevel.PRIVATE )
@Slf4j
public final class AuthTokenAccessor
{
    /**
     * The {@link AuthTokenFacade} instance.
     */
    @Getter
    @Nonnull
    private static AuthTokenFacade authTokenFacade = new DefaultAuthTokenFacade();

    /**
     * Global fallback {@link AuthToken}. By default, no fallback is used, i.e., the fallback is {@code null}. A global
     * fallback can be useful to ensure a safe fallback or to ease testing with a mocked token.
     */
    @Getter
    @Setter
    @Nullable
    private static Supplier<AuthToken> fallbackToken = null;

    /**
     * Replaces the default {@link AuthTokenFacade} instance.
     *
     * @param requestFacade
     *            An instance of {@link AuthTokenFacade}. Use {@code null} to reset the facade.
     */
    public static void setAuthTokenFacade( @Nullable final AuthTokenFacade requestFacade )
    {
        if( requestFacade == null ) {
            AuthTokenAccessor.authTokenFacade =
                new DefaultAuthTokenFacade(
                    OAuth2TokenServiceCache.create(),
                    DefaultAuthTokenFacade.loadOauth2Validators());
        } else {
            AuthTokenAccessor.authTokenFacade = requestFacade;
        }
    }

    /**
     * Returns the current {@link AuthToken}.
     *
     * @return The current {@link AuthToken}.
     *
     * @throws AuthTokenAccessException
     *             If there is an issue while trying to access the {@link AuthToken}. For instance, an {@link AuthToken}
     *             is not available if no request is available or the request does not contain an "Authorization"
     *             header.
     */
    @Nonnull
    public static AuthToken getCurrentToken()
        throws AuthTokenAccessException
    {
        return tryGetCurrentToken().getOrElseThrow(failure -> {
            if( failure instanceof AuthTokenAccessException ) {
                throw (AuthTokenAccessException) failure;
            } else {
                throw new AuthTokenAccessException("Failed to get current authorization token.", failure);
            }
        });
    }

    /**
     * Returns a {@link Try} of the current {@link AuthToken}, or, if the {@link Try} is a failure, the global fallback.
     * An {@link AuthToken} is not available if no request is available or the request does not contain an
     * "Authorization" header.
     *
     * @return A {@link Try} of the current {@link AuthToken}
     */
    @Nonnull
    public static Try<AuthToken> tryGetCurrentToken()
    {
        final Try<AuthToken> authTokenTry = authTokenFacade.tryGetCurrentToken();

        if( authTokenTry.isFailure() && fallbackToken != null ) {
            return authTokenTry.recover(failure -> {
                final AuthToken fallback = fallbackToken.get();
                log.warn("Recovering with fallback token: {}.", fallback, failure);
                return fallback;
            });
        }

        return authTokenTry;
    }

    /**
     * Retrieves a validated authentication token from the bound XSUAA instance.
     *
     * @return A {@link Try} of the XSUAA authentication token.
     *
     * @throws TokenRequestFailedException
     *             If no XSUAA instance was bound or the communication with the service failed.
     */
    @Nonnull
    public static AuthToken getXsuaaServiceToken()
        throws TokenRequestFailedException
    {
        return tryGetXsuaaServiceToken().getOrElseThrow(throwable -> {
            if( throwable instanceof TokenRequestFailedException ) {
                throw (TokenRequestFailedException) throwable;
            } else {
                throw new TokenRequestFailedException("Failed to get XSUAA service token.", throwable);
            }
        });
    }

    /**
     * Retrieves a validated authentication token from the bound XSUAA instance.
     *
     * @return A {@link Try} of the XSUAA authentication token.
     */
    @Nonnull
    public static Try<AuthToken> tryGetXsuaaServiceToken()
    {
        return authTokenFacade.tryGetXsuaaServiceToken();
    }

    /**
     * Returns a {@link Future} that resolves to a refresh token.
     * 
     * @param jwt
     *            The decoded JWT instance to resolve a refresh token for.
     *
     * @return A {@link Future} that resolves to a refresh token.
     * @throws TokenRequestFailedException
     *             When the token request failed.
     * @throws TokenRequestDeniedException
     *             When the token request was denied.
     */
    @Nonnull
    public static Future<String> getRefreshToken( @Nonnull final DecodedJWT jwt )
    {
        return authTokenFacade.getRefreshToken(jwt);
    }

    /**
     * Execute the given {@link Callable} with a given token.
     *
     * @param authToken
     *            The token to execute with.
     * @param callable
     *            The callable to execute.
     *
     * @param <T>
     *            The type of the callable.
     *
     * @return The value computed by the callable.
     *
     * @throws ThreadContextExecutionException
     *             If there is an issue while running the code with the token.
     */
    @Nullable
    public static <T> T executeWithAuthToken( @Nonnull final AuthToken authToken, @Nonnull final Callable<T> callable )
        throws ThreadContextExecutionException
    {
        return new ThreadContextExecutor()
            .withThreadContext(new DefaultThreadContext())
            .withListeners(new AuthTokenThreadContextListener(authToken))
            .execute(callable);
    }

    /**
     * Execute the given {@link Executable} with a given token.
     *
     * @param authToken
     *            The token to execute with.
     * @param executable
     *            The executable to execute.
     *
     * @throws ThreadContextExecutionException
     *             If there is an issue while running the code with the token.
     */
    public static void executeWithAuthToken( @Nonnull final AuthToken authToken, @Nonnull final Executable executable )
        throws ThreadContextExecutionException
    {
        executeWithAuthToken(authToken, () -> {
            executable.execute();
            return null;
        });
    }

    /**
     * Execute the given {@link Callable}, using the given token as fallback if there is no other token available.
     *
     * @param fallbackAuthToken
     *            The token to fall back to.
     * @param callable
     *            The callable to execute.
     *
     * @param <T>
     *            The type of the callable.
     *
     * @return The value computed by the callable.
     *
     * @throws ThreadContextExecutionException
     *             If there is an issue while running the code with the token.
     */
    @Nullable
    public static <T> T executeWithFallbackAuthToken(
        @Nonnull final Supplier<AuthToken> fallbackAuthToken,
        @Nonnull final Callable<T> callable )
        throws ThreadContextExecutionException
    {
        final Try<AuthToken> tokenTry = tryGetCurrentToken();

        if( tokenTry.isSuccess() ) {
            try {
                return callable.call();
            }
            catch( final ThreadContextExecutionException e ) {
                throw e;
            }
            catch( final Exception e ) {
                throw new ThreadContextExecutionException(e);
            }
        }

        return executeWithAuthToken(fallbackAuthToken.get(), callable);
    }

    /**
     * Execute the given {@link Executable}, using the given token as fallback if there is no other token available.
     *
     * @param fallbackAuthToken
     *            The token to fall back to.
     * @param executable
     *            The executable to execute.
     *
     * @throws ThreadContextExecutionException
     *             If there is an issue while running the code with the token.
     */
    public static void executeWithFallbackAuthToken(
        @Nonnull final Supplier<AuthToken> fallbackAuthToken,
        @Nonnull final Executable executable )
        throws ThreadContextExecutionException
    {
        executeWithFallbackAuthToken(fallbackAuthToken, () -> {
            executable.execute();
            return null;
        });
    }
}
