/*
 * Copyright (c) 2022 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Collection;
import java.util.Collections;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.auth0.jwt.JWT;
import com.auth0.jwt.JWTCreator;
import com.auth0.jwt.algorithms.Algorithm;
import com.auth0.jwt.interfaces.DecodedJWT;
import com.google.gson.JsonElement;
import com.sap.cloud.sdk.cloudplatform.CloudPlatformAccessor;
import com.sap.cloud.sdk.cloudplatform.ScpCfCloudPlatform;
import com.sap.cloud.sdk.cloudplatform.exception.CloudPlatformException;

import lombok.RequiredArgsConstructor;

/**
 * Utility class to manage decoding and encoding of JWT.
 */
public class AuthTokenBuilder
{
    /**
     * Fluent helper to chain operations. For internal use only.
     */
    public interface FluentHelperBuild
    {
        /**
         * Builds the {@link AuthToken}.
         *
         * @return The constructed {@link AuthToken}.
         */
        @Nonnull
        AuthToken build();
    }

    /**
     * Fluent helper to chain operations. For internal use only.
     */
    @RequiredArgsConstructor
    public static class FluentHelperRefreshToken implements FluentHelperBuild
    {
        @Nonnull
        private final String encodedJwt;

        @Nullable
        private String refreshToken;

        /**
         * Specifies to use the given refresh token to refresh the encoded JWT, if expired.
         *
         * @param refreshToken
         *            The refresh token to be used to refresh the JWT, if necessary.
         *
         * @return A fluent helper to chain operations.
         */
        @Nonnull
        public FluentHelperBuild refreshToken( @Nonnull final String refreshToken )
        {
            this.refreshToken = refreshToken;
            return this;
        }

        @Override
        @Nonnull
        public AuthToken build()
        {
            return new AuthTokenDecoder().decodeAndValidate(encodedJwt, refreshToken);
        }
    }

    public static class FluentHelperUnsignedJwt
    {
        @Nullable
        private String zid;

        @Nullable
        private String iss;

        @Nullable
        private Collection<String> audiences;

        /**
         * Construct the URL of subscriber XSUAA endpoint as artificial JWT issuer.
         *
         * @param subdomain
         *            The subdomain of the tenant.
         *
         * @return The constructed URI of subscriber XSUAA endpoint.
         *
         * @throws CloudPlatformException
         *             If there is an issue requesting the token.
         */
        @Nonnull
        private URI getIssuerUri( @Nonnull final String subdomain )
            throws CloudPlatformException
        {
            final JsonElement url = ScpCfCloudPlatform.getInstanceOrThrow().getXsuaaServiceCredentials().get("url");

            if( url == null || !url.isJsonPrimitive() ) {
                throw new CloudPlatformException(
                    "Failed to get XSUAA service URI: no 'url' present in service credentials.");
            }

            try {
                final URI uri = new URI(url.getAsString());
                return new URI(
                    uri.getScheme(),
                    null,
                    subdomain + uri.getHost().substring(uri.getHost().indexOf(".")),
                    uri.getPort(),
                    uri.getPath(),
                    null,
                    null);
            }
            catch( final URISyntaxException e ) {
                throw new CloudPlatformException(e);
            }
        }

        /**
         * A non-validated JWT is constructed based on tenant information ({@code xsappname} and issuer URL of
         * subscriber in {@code VCAP_SERVICES} environment variable). This artificial JWT can be used in the
         * {@code execute} methods of {@link AuthTokenAccessor} so that operations in this context are executed on
         * behalf of the subscriber tenant provided. The generated JWT contains default values for issuer URL and
         * audience.
         *
         * @param tenantId
         *            The tenant or zone identifier to be used and acted on behalf of.
         * @param subdomain
         *            The subdomain of the tenant/zone to act on behalf of.
         *
         * @return A fluent helper to chain operations.
         *
         * @throws CloudPlatformException
         *             If there is an issue retrieving the required tenant information.
         */
        @Nonnull
        public FluentHelperBuild ofTenant( @Nonnull final String tenantId, @Nonnull final String subdomain )
            throws CloudPlatformException
        {
            final String xsAppName = ((ScpCfCloudPlatform) CloudPlatformAccessor.getCloudPlatform()).getXsAppName();
            return ofTenant(tenantId, getIssuerUri(subdomain), Collections.singletonList(xsAppName));
        }

        /**
         * A non-validated JWT is constructed based on tenant information. This artificial JWT can be used in the
         * {@code execute} methods of {@link AuthTokenAccessor} so that operations in this context are executed on
         * behalf of the subscriber tenant provided.
         *
         * @param tenantId
         *            The tenant identifier to be used and acted on behalf of.
         * @param issuerUri
         *            The issuer URL of the subscriber OAuth2 endpoint.
         * @param audiences
         *            A customizable array of JWT audience entries. Adding the application name is recommended.
         *
         * @return A fluent helper to chain operations.
         */
        @Nonnull
        public FluentHelperBuild ofTenant(
            @Nonnull final String tenantId,
            @Nonnull final URI issuerUri,
            @Nonnull final Collection<String> audiences )
        {
            zid = tenantId;
            iss = issuerUri.toString();
            this.audiences = audiences;
            return fluentHelperBuild();
        }

        private FluentHelperBuild fluentHelperBuild()
        {
            return () -> {
                final JWTCreator.Builder jwtBuilder = JWT.create();

                if( zid != null ) {
                    jwtBuilder.withClaim("zid", zid);
                }
                if( iss != null ) {
                    jwtBuilder.withClaim("iss", iss);
                }

                if( audiences != null ) {
                    jwtBuilder.withAudience(audiences.toArray(new String[0]));
                }

                final String encodedJwt = jwtBuilder.sign(Algorithm.none());

                final DecodedJWT decodedJwt = JWT.decode(encodedJwt);
                return new AuthToken(decodedJwt);
            };
        }
    }

    /**
     * Use a {@link DecodedJWT} for constructing the {@link AuthToken}.
     *
     * @param decodedJwt
     *            The decoded JWT. The given value will be used without any validation.
     *
     * @return A fluent helper to chain operations.
     */
    @Nonnull
    public FluentHelperBuild decodedJwt( @Nonnull final DecodedJWT decodedJwt )
    {
        return () -> new AuthToken(decodedJwt);
    }

    /**
     * Use an encoded JWT for constructing the {@link AuthToken}.
     *
     * @param encodedJwt
     *            The encoded JWT. The given value will be validated and decoded.
     *
     * @return A fluent helper to chain operations.
     */
    @Nonnull
    public FluentHelperRefreshToken encodedJwt( @Nonnull final String encodedJwt )
    {
        return new FluentHelperRefreshToken(encodedJwt);
    }

    /**
     * Use an non-validated, unsigned JWT for constructing the {@link AuthToken}. This artificial JWT can be used in the
     * {@code execute} methods of {@link AuthTokenAccessor}.
     *
     * @return A fluent helper to chain operations.
     */
    @Nonnull
    public FluentHelperUnsignedJwt unsignedJwt()
    {
        return new FluentHelperUnsignedJwt();
    }
}
