/*
 * Copyright (c) 2022 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security;

import java.util.concurrent.Future;

import javax.annotation.Nonnull;

import com.auth0.jwt.interfaces.DecodedJWT;

import io.vavr.control.Try;

/**
 * Facade for retrieving the current {@link AuthToken}.
 */
public interface AuthTokenFacade
{
    /**
     * Returns a {@link Try} of the current {@link AuthToken}. An {@link AuthToken} is not available if no request is
     * available or the request does not contain an "Authorization" header.
     *
     * @return A {@link Try} of the current {@link AuthToken}.
     */
    @Nonnull
    Try<AuthToken> tryGetCurrentToken();

    /**
     * Retrieves a validated authentication token from the bound XSUAA instance.
     *
     * @return A {@link Try} of the XSUAA authentication token.
     */
    @Nonnull
    Try<AuthToken> tryGetXsuaaServiceToken();

    /**
     * Returns a {@link Future} that resolves to a refresh token.
     *
     * @param jwt
     *            The decoded JWT instance to resolve a refresh token for.
     * @return A {@link Future} that resolves to a refresh token.
     * @throws TokenRequestFailedException
     *             When the token request failed.
     * @throws TokenRequestDeniedException
     *             When the token request was denied.
     */
    @Nonnull
    Future<String> getRefreshToken( @Nonnull final DecodedJWT jwt );
}
