/*
 * Copyright (c) 2022 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.sap.cloud.sdk.cloudplatform.thread.Property;
import com.sap.cloud.sdk.cloudplatform.thread.ThreadContext;
import com.sap.cloud.sdk.cloudplatform.thread.ThreadContextListener;

import io.vavr.control.Try;

/**
 * Implementation of {@link ThreadContextListener} that ensures the correct initialization of {@link AuthToken}s when
 * working with non-container managed threads on all supported Cloud platforms.
 */
public class AuthTokenThreadContextListener implements ThreadContextListener
{
    /**
     * The ThreadContext key.
     */
    public static final String PROPERTY_AUTH_TOKEN = AuthTokenThreadContextListener.class.getName() + ":authToken";

    /**
     * The {@link AuthToken} to be used by this listener.
     */
    @Nullable
    private final AuthToken authToken;

    /**
     * Default constructor.
     */
    public AuthTokenThreadContextListener()
    {
        authToken = null;
    }

    /**
     * Constructor for providing a {@link AuthToken} to be returned by this listener.
     *
     * @param authToken
     *            The {@link AuthToken} to be used by this listener.
     */
    public AuthTokenThreadContextListener( @Nonnull final AuthToken authToken )
    {
        this.authToken = authToken;
    }

    @Override
    public int getPriority()
    {
        return DefaultPriorities.AUTH_TOKEN_LISTENER;
    }

    @Override
    public
        void
        afterInitialize( @Nonnull final ThreadContext threadContext, @Nullable final ThreadContext parentThreadContext )
    {
        final Try<AuthToken> authTokenTry;

        if( authToken != null ) {
            authTokenTry = Try.success(authToken);
        } else {
            if( parentThreadContext != null ) {
                authTokenTry =
                    parentThreadContext
                        .<AuthToken> getProperty(PROPERTY_AUTH_TOKEN)
                        .map(Property::getValue)
                        .orElse(AuthTokenAccessor::tryGetCurrentToken);
            } else {
                authTokenTry = AuthTokenAccessor.tryGetCurrentToken();
            }
        }

        threadContext.setPropertyIfAbsent(PROPERTY_AUTH_TOKEN, Property.ofTry(authTokenTry));
    }

    @Override
    public
        void
        beforeDestroy( @Nonnull final ThreadContext threadContext, @Nullable final ThreadContext parentThreadContext )
    {
        threadContext.removeProperty(PROPERTY_AUTH_TOKEN);
    }
}
