/*
 * Copyright (c) 2022 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.sap.cloud.sdk.cloudplatform.thread.Property;
import com.sap.cloud.sdk.cloudplatform.thread.ThreadContext;
import com.sap.cloud.sdk.cloudplatform.thread.ThreadContextListener;

import io.vavr.control.Try;

/**
 * {@link ThreadContextListener} implementation reading the Basic Authentication Header from the incoming request and
 * storing it in the current {@link ThreadContext}.
 */
public class BasicAuthenticationThreadContextListener implements ThreadContextListener
{
    /**
     * The key of the stored BasicCredentials in the current {@link ThreadContext}.
     */
    public static final String PROPERTY_BASIC_AUTH_HEADER =
        BasicAuthenticationThreadContextListener.class.getName() + ":basicAuthHeader";

    @Override
    public int getPriority()
    {
        return DefaultPriorities.BASIC_AUTH_LISTENER;
    }

    @Override
    public
        void
        afterInitialize( @Nonnull final ThreadContext threadContext, @Nullable final ThreadContext parentThreadContext )
    {
        final Try<BasicCredentials> basicCredentialsTry;

        if( parentThreadContext != null ) {
            basicCredentialsTry =
                parentThreadContext
                    .<BasicCredentials> getProperty(PROPERTY_BASIC_AUTH_HEADER)
                    .map(Property::getValue)
                    .orElse(BasicAuthenticationAccessor::tryGetCurrentBasicCredentials);
        } else {
            basicCredentialsTry = BasicAuthenticationAccessor.tryGetCurrentBasicCredentials();
        }

        threadContext.setPropertyIfAbsent(PROPERTY_BASIC_AUTH_HEADER, Property.ofTry(basicCredentialsTry));
    }

    @Override
    public
        void
        beforeDestroy( @Nonnull final ThreadContext threadContext, @Nullable final ThreadContext parentThreadContext )
    {
        threadContext.removeProperty(PROPERTY_BASIC_AUTH_HEADER);
    }
}
