/*
 * Copyright (c) 2022 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security;

import static com.sap.cloud.sdk.cloudplatform.resilience.ResilienceConfiguration.CircuitBreakerConfiguration;
import static com.sap.cloud.sdk.cloudplatform.resilience.ResilienceConfiguration.TimeLimiterConfiguration;

import java.time.Duration;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.Callable;
import java.util.concurrent.Future;
import java.util.concurrent.FutureTask;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.auth0.jwt.interfaces.DecodedJWT;
import com.google.common.collect.Sets;
import com.google.gson.Gson;
import com.sap.cloud.sdk.cloudplatform.ScpCfCloudPlatform;
import com.sap.cloud.sdk.cloudplatform.requestheader.RequestHeaderAccessor;
import com.sap.cloud.sdk.cloudplatform.resilience.ResilienceConfiguration;
import com.sap.cloud.sdk.cloudplatform.resilience.ResilienceDecorator;
import com.sap.cloud.sdk.cloudplatform.resilience.ResilienceIsolationMode;
import com.sap.cloud.sdk.cloudplatform.security.exception.TokenRequestDeniedException;
import com.sap.cloud.sdk.cloudplatform.security.exception.TokenRequestFailedException;
import com.sap.cloud.sdk.cloudplatform.thread.Property;
import com.sap.cloud.sdk.cloudplatform.thread.ThreadContextAccessor;
import com.sap.cloud.security.config.OAuth2ServiceConfiguration;
import com.sap.cloud.security.config.Service;
import com.sap.cloud.security.config.cf.CFConstants;
import com.sap.cloud.security.config.cf.CFEnvironment;
import com.sap.cloud.security.token.Token;
import com.sap.cloud.security.token.validation.CombiningValidator;
import com.sap.cloud.security.token.validation.validators.JwtValidatorBuilder;
import com.sap.cloud.security.xsuaa.client.OAuth2TokenService;
import com.sap.cloud.security.xsuaa.tokenflows.XsuaaTokenFlows;

import io.vavr.control.Option;
import io.vavr.control.Try;
import lombok.AccessLevel;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

/**
 * Facade for retrieving the current {@link AuthToken}.
 */
@Slf4j
@RequiredArgsConstructor( access = AccessLevel.PACKAGE )
public class DefaultAuthTokenFacade implements AuthTokenFacade
{
    private static final String MESSAGE_INVALID_REFRESH_TOKEN =
        "Failed to get access token: no valid refresh token found in response of user token flow. Please make sure to correctly bind your application to an OAuth2 compatible service instance.";

    @Nonnull
    static final OAuth2TokenServiceCache DEFAULT_TOKEN_SERVICE_CACHE = OAuth2TokenServiceCache.create();

    @Nonnull
    static final List<CombiningValidator<Token>> DEFAULT_VALIDATORS = loadOauth2Validators();

    @Nonnull
    private final OAuth2TokenServiceCache tokenServiceCache;

    @Nonnull
    private final List<CombiningValidator<Token>> tokenValidators;

    /**
     * Default constructor.
     */
    public DefaultAuthTokenFacade()
    {
        this(DEFAULT_TOKEN_SERVICE_CACHE, DEFAULT_VALIDATORS);
    }

    /**
     * Create a new {@code DefaultAuthTokenFacade} with the given token service and an
     * {@link OAuth2ServiceConfiguration}. The latter will be used to construct a validator for auth token validation.
     *
     * @param tokenService
     *            The {@link OAuth2TokenService} to be used.
     * @param oauth2Configuration
     *            The OAuth2 configuration to be used.
     */
    public DefaultAuthTokenFacade(
        @Nonnull final OAuth2TokenService tokenService,
        @Nullable final OAuth2ServiceConfiguration oauth2Configuration )
    {
        this(
            OAuth2TokenServiceCache.single(tokenService),
            Option
                .of(oauth2Configuration)
                .onEmpty(() -> log.warn("AuthTokenFacade instantiated without an OAuth2 configuration."))
                .map(DefaultAuthTokenFacade::getJwtValidator)
                .map(Collections::singletonList)
                .getOrElse(Collections::emptyList));
    }

    @Nonnull
    @Override
    public Try<AuthToken> tryGetCurrentToken()
    {
        return ThreadContextAccessor
            .tryGetCurrentContext()
            .flatMap(context -> context.<AuthToken> getProperty(AuthTokenThreadContextListener.PROPERTY_AUTH_TOKEN))
            .map(Property::getValue)
            .orElse(
                () -> RequestHeaderAccessor.tryGetHeaderContainer().flatMap(getAuthTokenDecoder()::decodeAndValidate));
    }

    @Nonnull
    @Override
    public Try<AuthToken> tryGetXsuaaServiceToken()
    {
        final AuthTokenDecoder authTokenDecoder = getAuthTokenDecoder();
        return Try.of(() -> new AuthTokenRequest(authTokenDecoder, tokenServiceCache).getXsuaaServiceToken());
    }

    @Nonnull
    @Override
    public Future<String> getRefreshToken( @Nonnull final DecodedJWT jwt )
    {
        final ResilienceConfiguration resilienceConfiguration =
            ResilienceConfiguration
                .of(DefaultAuthTokenFacade.class)
                .isolationMode(ResilienceIsolationMode.NO_ISOLATION)
                .timeLimiterConfiguration(TimeLimiterConfiguration.of().timeoutDuration(Duration.ofSeconds(6)))
                .circuitBreakerConfiguration(CircuitBreakerConfiguration.of().waitDuration(Duration.ofSeconds(6)));

        final Callable<String> tokenCallable =
            ResilienceDecorator.decorateCallable(() -> sendTokenRequestAndParseResponse(jwt), resilienceConfiguration);

        final FutureTask<String> tokenFuture = new FutureTask<>(tokenCallable);
        tokenFuture.run();
        return tokenFuture;
    }

    @Nonnull
    private String sendTokenRequestAndParseResponse( @Nonnull final DecodedJWT jwt )
        throws TokenRequestDeniedException,
            TokenRequestFailedException
    {
        final String authorizationBearer = jwt.getToken();
        final XsuaaTokenFlows tokenFlows =
            OAuth2ServiceProvider
                .builder()
                .tokenServiceCache(tokenServiceCache)
                .staticAccessToken(jwt)
                .build()
                .getXsuaaTokenFlows();

        return Try
            .of(() -> tokenFlows.userTokenFlow().token(authorizationBearer).execute().getRefreshToken())
            .onFailure(e -> log.debug("User token request failed for auth token {}.", authorizationBearer, e))
            .filter(Objects::nonNull, () -> new TokenRequestFailedException(MESSAGE_INVALID_REFRESH_TOKEN))
            .getOrElseThrow(e -> new TokenRequestFailedException("Refresh JWT request failed", e));
    }

    @SuppressWarnings( "deprecation" )
    @Nonnull
    static CombiningValidator<Token> getJwtValidator( @Nonnull final OAuth2ServiceConfiguration configuration )
    {
        final CombiningValidator<Token> result = JwtValidatorBuilder.getInstance(configuration).build();

        // see backlog item to remove this workaround: CLOUDECOSYSTEM-9585
        final String workaroundClass = "com.sap.cloud.security.token.validation.validators.JwtIssuerValidator";
        if( result.getValidators().removeIf(validator -> workaroundClass.equals(validator.getClass().getName())) ) {
            result.getValidators().add(new CustomJwtIssuerValidator(configuration));
        }

        return result;
    }

    @Nonnull
    static List<CombiningValidator<Token>> loadOauth2Validators()
    {
        final List<OAuth2ServiceConfiguration> oauth2Configurations = loadOauth2ServiceConfigurations();
        if( oauth2Configurations.isEmpty() ) {
            log.debug("No OAuth2 validators were registered since no configuration could be loaded.");
        }

        return oauth2Configurations
            .stream()
            .map(conf -> Try.of(() -> DefaultAuthTokenFacade.getJwtValidator(conf)))
            .filter(validator -> validator.onFailure(e -> log.warn("Failed to load validator.", e)).isSuccess())
            .map(Try::get)
            .collect(Collectors.toList());
    }

    @Nonnull
    private static List<OAuth2ServiceConfiguration> loadOauth2ServiceConfigurations()
    {
        // resolve environment properties for CloudFoundry, handle potential parsing errors
        final Try<CFEnvironment> environment = Try.of(() -> {
            final ScpCfCloudPlatform platform = ScpCfCloudPlatform.getInstanceOrThrow();
            return CFEnvironment.getInstance(k -> getEnvironmentVariable(platform, k), System::getProperty);
        });

        if( environment.isSuccess() ) {

            // read all configurations from environment, consider all potential service plans
            final List<OAuth2ServiceConfiguration> configurations =
                Sets
                    .cartesianProduct(EnumSet.allOf(Service.class), EnumSet.allOf(CFConstants.Plan.class))
                    .stream()
                    .map(p -> environment.get().loadForServicePlan((Service) p.get(0), (CFConstants.Plan) p.get(1)))
                    .filter(Objects::nonNull)
                    .peek(
                        c -> log
                            .debug(
                                "Found {} service binding with plan {}, client id {} and application id {}.",
                                c.getService().getCFName(),
                                c.getProperty(CFConstants.SERVICE_PLAN),
                                c.getClientId(),
                                c.getProperty(CFConstants.XSUAA.APP_ID)))
                    .collect(Collectors.toList());

            if( configurations.isEmpty() ) {
                log.warn("Could not find any OAuth2 based service bindings.");
            }
            return configurations;
        }
        log.error("Failed to read environment data for OAuth2 based configurations.", environment.getCause());
        return Collections.emptyList();
    }

    @Nonnull
    private static
        String
        getEnvironmentVariable( @Nonnull final ScpCfCloudPlatform platform, @Nonnull final String variableName )
    {
        /*
        If the Security Library requests the "VCAP_SERVICES" environment variable, we want to return **all** service
        bindings, including those from the local file system (in the K8s case).
        Therefore, instead of simply returning the result of `platform.getEnvironmentVariable(variableName).getOrNull()`
        we need to make sure to return the result of our `ServiceBindingLoader` (as configured in the ScpCfCloudPlatform).
         */
        if( "VCAP_SERVICES".equalsIgnoreCase(variableName) ) {
            final Gson gson = new Gson();
            return Try.of(platform::getVcapServices).map(gson::toJson).getOrNull();
        }

        // For all other environment variables, we want to return the regular result of our environment variable reader.
        return platform.getEnvironmentVariable(variableName).getOrNull();
    }

    AuthTokenDecoder getAuthTokenDecoder()
    {
        return new AuthTokenDecoder(tokenServiceCache, tokenValidators);
    }
}
