/*
 * Copyright (c) 2022 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Collections;
import java.util.Enumeration;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.Nonnull;
import javax.servlet.http.HttpServletRequest;

import org.apache.http.HttpHeaders;

import com.sap.cloud.sdk.cloudplatform.requestheader.RequestHeaderAccessor;
import com.sap.cloud.sdk.cloudplatform.security.exception.BasicAuthenticationAccessException;
import com.sap.cloud.sdk.cloudplatform.servlet.RequestAccessor;
import com.sap.cloud.sdk.cloudplatform.thread.Property;
import com.sap.cloud.sdk.cloudplatform.thread.ThreadContext;
import com.sap.cloud.sdk.cloudplatform.thread.ThreadContextAccessor;

import io.vavr.control.Try;

/**
 * Default implementation of the {@link BasicAuthenticationAccessor} reading the username and password from the current
 * {@link ThreadContext} or, after that, from the currently incoming request and returning it as a
 * {@link BasicCredentials} object.
 */
public class DefaultBasicAuthenticationFacade implements BasicAuthenticationFacade
{
    /**
     * Regex pattern to match the basic authentication header value.
     * <p>
     * Noteworthy points:
     * <ul>
     * <li>the "Basic" prefix is case insensitive</li>
     * <li>the base64 encoded actual value can be wrapped in arbitrarily many whitespaces</li>
     * <li>the actual value will be stored in the first (and only) matching group of the regex</li>
     * </ul>
     */
    private static final Pattern CASE_INSENSITIVE_BASIC_PREFIX_MATCH =
        Pattern.compile("^(?i)basic(?-i) \\s*([A-Za-z0-9+/=]+)\\s*$");

    @Nonnull
    @Override
    public Try<BasicCredentials> tryGetBasicCredentials()
    {
        final List<Throwable> suppressed = new ArrayList<>();

        final Try<BasicCredentials> result =
            readBasicCredentialsFromContext()
                .onFailure(suppressed::add)
                .orElse(this::extractBasicCredentialsFromHeader)
                .onFailure(suppressed::add)
                .orElse(this::extractBasicCredentialsFromRequest)
                .onFailure(suppressed::add);

        if( result.isFailure() ) {
            final Exception exception = new BasicAuthenticationAccessException("Unable to resolve basic credentials.");
            suppressed.forEach(exception::addSuppressed);
            return Try.failure(exception);
        }
        return result;
    }

    private Try<BasicCredentials> readBasicCredentialsFromContext()
    {
        return ThreadContextAccessor
            .tryGetCurrentContext()
            .flatMap(this::extractBasicCredentialsProperty)
            .map(Property::getValue);
    }

    private Try<Property<BasicCredentials>> extractBasicCredentialsProperty( final ThreadContext context )
    {
        return context.getProperty(BasicAuthenticationThreadContextListener.PROPERTY_BASIC_AUTH_HEADER);
    }

    private Try<BasicCredentials> extractBasicCredentialsFromHeader()
    {
        return RequestHeaderAccessor
            .tryGetHeaderContainer()
            .map(container -> new ArrayList<>(container.getHeaderValues(HttpHeaders.AUTHORIZATION)))
            .map(this::selectBasicAuthenticationHeader)
            .map(this::extractBasicHeaderValue)
            .map(this::decodeBasicCredentials);
    }

    private Try<BasicCredentials> extractBasicCredentialsFromRequest()
    {
        return RequestAccessor
            .tryGetCurrentRequest()
            .map(this::extractAuthorizationHeaders)
            .map(this::selectBasicAuthenticationHeader)
            .map(this::extractBasicHeaderValue)
            .map(this::decodeBasicCredentials);
    }

    private List<String> extractAuthorizationHeaders( final HttpServletRequest request )
    {
        final Enumeration<String> headers = request.getHeaders(HttpHeaders.AUTHORIZATION);
        if( headers == null || !headers.hasMoreElements() ) {
            throw new BasicAuthenticationAccessException(
                "Received no '" + HttpHeaders.AUTHORIZATION + "' headers with the request.");
        }
        return Collections.list(headers);
    }

    private String selectBasicAuthenticationHeader( final List<String> allAuthenticationHeader )
    {
        if( allAuthenticationHeader.isEmpty() ) {
            throw new BasicAuthenticationAccessException(
                "Received an '" + HttpHeaders.AUTHORIZATION + "' header without a value.");
        }

        if( allAuthenticationHeader.size() > 1 ) {
            throw new BasicAuthenticationAccessException(
                "Received multiple '"
                    + HttpHeaders.AUTHORIZATION
                    + "' headers with the request, but the specification allows at most one.");
        }
        return allAuthenticationHeader.get(0);
    }

    private String extractBasicHeaderValue( final CharSequence completeHeader )
    {
        final Matcher match = CASE_INSENSITIVE_BASIC_PREFIX_MATCH.matcher(completeHeader);
        if( !match.matches() ) {
            throw new BasicAuthenticationAccessException(
                "The '" + HttpHeaders.AUTHORIZATION + "' header did not contain a Basic Authentication header field.");
        }
        return match.group(1);
    }

    private BasicCredentials decodeBasicCredentials( final String base64Credentials )
    {
        final String[] credentials =
            new String(Base64.getDecoder().decode(base64Credentials), StandardCharsets.UTF_8).split(":");

        return new BasicCredentials(credentials[0], credentials[1]);
    }
}
