/*
 * Copyright (c) 2022 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security;

import static com.sap.cloud.security.config.Service.XSUAA;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.UnaryOperator;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.auth0.jwt.interfaces.DecodedJWT;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.CloudPlatform;
import com.sap.cloud.sdk.cloudplatform.CloudPlatformAccessor;
import com.sap.cloud.sdk.cloudplatform.ScpCfCloudPlatform;
import com.sap.cloud.sdk.cloudplatform.exception.CloudPlatformException;
import com.sap.cloud.security.config.ClientIdentity;
import com.sap.cloud.security.config.OAuth2ServiceConfiguration;
import com.sap.cloud.security.config.cf.CFConstants;
import com.sap.cloud.security.config.cf.CFEnvironment;
import com.sap.cloud.security.xsuaa.client.OAuth2ServiceEndpointsProvider;
import com.sap.cloud.security.xsuaa.client.OAuth2TokenService;
import com.sap.cloud.security.xsuaa.client.XsuaaDefaultEndpoints;
import com.sap.cloud.security.xsuaa.tokenflows.XsuaaTokenFlows;

import io.vavr.Lazy;
import lombok.Builder;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@Value
class DefaultOAuth2ServiceProvider implements OAuth2ServiceProvider
{
    @Nonnull
    CloudPlatform platform;

    @Nonnull
    OAuth2TokenServiceCache tokenServiceCache;

    @Nullable
    OAuth2TokenService staticTokenService;

    @Nullable
    ClientIdentity staticClientIdentity;

    @Nullable
    OAuth2ServiceEndpointsProvider staticEndpointsProvider;

    @Nullable
    DecodedJWT staticAccessToken;

    @Nonnull
    Lazy<CFEnvironment> cfEnvironment = Lazy.of(this::createEnvironment);

    @Builder
    private DefaultOAuth2ServiceProvider(
        @Nullable final OAuth2TokenServiceCache tokenServiceCache,
        @Nullable final CloudPlatform platform,
        @Nullable final OAuth2TokenService staticTokenService,
        @Nullable final DecodedJWT staticAccessToken,
        @Nullable final Credentials staticCredentials,
        @Nullable final OAuth2ServiceSettings serviceSettings )
    {
        this.staticClientIdentity = staticCredentials == null ? null : getClientIdentity(staticCredentials);
        this.staticEndpointsProvider = serviceSettings == null ? null : serviceSettings.toOAuth2Endpoints();
        this.platform = platform != null ? platform : CloudPlatformAccessor.getCloudPlatform();
        this.tokenServiceCache = tokenServiceCache != null ? tokenServiceCache : OAuth2TokenServiceCache.create();

        this.staticAccessToken = staticAccessToken;
        this.staticTokenService = staticTokenService;
    }

    @Nonnull
    private static ClientIdentity getClientIdentity( @Nonnull final Credentials credentials )
    {
        if( credentials instanceof ClientCredentials ) {
            return new com.sap.cloud.security.config.ClientCredentials(
                ((ClientCredentials) credentials).getClientId(),
                ((ClientCredentials) credentials).getClientSecret());
        }
        if( credentials instanceof ClientCertificate ) {
            return new com.sap.cloud.security.config.ClientCertificate(
                ((ClientCertificate) credentials).getCertificate(),
                ((ClientCertificate) credentials).getKey(),
                ((ClientCertificate) credentials).getClientId());
        }
        throw new IllegalStateException(
            "Unsupported credentials type for authenticating against OAuth2 endpoint: "
                + credentials.getClass().getSimpleName());
    }

    @Nonnull
    OAuth2ServiceConfiguration getCurrentServiceConfiguration()
    {
        final CFEnvironment environment = cfEnvironment.get();
        if( environment.getXsuaaConfiguration() == null && environment.getIasConfiguration() == null ) {
            throw new CloudPlatformException("No XSUAA or IAS bindings found.");
        }
        final DecodedJWT jwt = staticAccessToken != null ? staticAccessToken : getCurrentAccessToken();
        final String plan = getCurrentXsuaaServicePlan(platform, jwt);

        // fallback logic
        if( plan == null ) {
            log
                .debug(
                    "Could not find any XSUAA service binding matching the current user access token. Falling back to default XSUAA binding.");
            return Objects.requireNonNull(environment.getXsuaaConfiguration(), "No XSUAA bindings found.");
        }

        // happy path
        final OAuth2ServiceConfiguration conf = environment.loadForServicePlan(XSUAA, CFConstants.Plan.from(plan));
        if( conf == null ) {
            throw new IllegalStateException("Unable to load XSUAA service information for plan " + plan);
        }
        return conf;
    }

    @Nullable
    private static
        String
        getCurrentXsuaaServicePlan( @Nonnull final CloudPlatform platform, @Nullable final DecodedJWT jwt )
    {
        final Map<String, List<JsonObject>> serviceBindings = getCurrentXsuaaServiceCredentials(platform, jwt);
        if( serviceBindings.size() > 1 ) {
            log.warn("Found more than one matching XSUAA service binding plan for provided token.");
        }
        return serviceBindings.isEmpty() ? null : serviceBindings.keySet().iterator().next();
    }

    @Nonnull
    private static
        Map<String, List<JsonObject>>
        getCurrentXsuaaServiceCredentials( @Nonnull final CloudPlatform platform, @Nullable final DecodedJWT jwt )
            throws CloudPlatformException
    {
        final Map<String, List<JsonObject>> result = castScpCfPlatform(platform).getXsuaaCredentialsByPlan(jwt);

        if( result.isEmpty() ) {
            if( jwt != null ) {
                log.debug("Unable to find matching XSUAA service binding for token: {}", jwt.getToken());
            } else {
                log.debug("Unable to find any XSUAA service binding.");
            }
        }
        return result;
    }

    @Nonnull
    private CFEnvironment createEnvironment()
    {
        final UnaryOperator<String> environmentOperator;
        if( platform instanceof ScpCfCloudPlatform ) {
            environmentOperator = k -> ((ScpCfCloudPlatform) platform).getEnvironmentVariable(k).getOrNull();
        } else {
            environmentOperator = System::getenv;
            log.warn("Provided platform object is not of type {}.", ScpCfCloudPlatform.class.getSimpleName());
        }
        return CFEnvironment.getInstance(environmentOperator, System::getProperty);
    }

    @Nonnull
    private static ScpCfCloudPlatform castScpCfPlatform( @Nonnull final CloudPlatform platform )
    {
        if( !(platform instanceof ScpCfCloudPlatform) ) {
            throw new CloudPlatformException(
                "This operation requires a platform instance of type: " + ScpCfCloudPlatform.class.getSimpleName());
        }
        return (ScpCfCloudPlatform) platform;
    }

    @Nullable
    private static DecodedJWT getCurrentAccessToken()
    {
        return AuthTokenAccessor
            .tryGetCurrentToken()
            .map(AuthToken::getJwt)
            .onFailure(e -> log.debug("Current user access token not found."))
            .getOrNull();
    }

    @Nonnull
    @Override
    public XsuaaTokenFlows getXsuaaTokenFlows()
    {
        // Only prepare current service configuration if the value will be used later on.
        OAuth2ServiceConfiguration currentServiceConfiguration = null;
        if( staticClientIdentity == null || staticEndpointsProvider == null ) {
            currentServiceConfiguration = getCurrentServiceConfiguration();
        }

        // Derive client identity object, either from static value or dynamically from current service configuration.
        ClientIdentity clientIdentity = staticClientIdentity;
        if( clientIdentity == null ) {
            clientIdentity = currentServiceConfiguration.getClientIdentity();
        }

        // Derive static token service object, either from static value or dynamically on behalf of client identity.
        OAuth2TokenService tokenService = staticTokenService;
        if( tokenService == null ) {
            tokenService = tokenServiceCache.getTokenService(clientIdentity);
        }

        // Derive service endpoints, either from static value or dynamically from current service configuration.
        OAuth2ServiceEndpointsProvider endpointsProvider = staticEndpointsProvider;
        if( endpointsProvider == null ) {
            endpointsProvider = new XsuaaDefaultEndpoints(currentServiceConfiguration);
        }

        return new XsuaaTokenFlows(tokenService, endpointsProvider, clientIdentity);
    }

    @SuppressWarnings( "unused" ) // used by Lombok
    static class DefaultOAuth2ServiceProviderBuilder implements OAuth2ServiceProvider.OAuth2ServiceProviderBuilder
    {
    }
}
