/*
 * Copyright (c) 2022 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security;

import java.util.Objects;
import java.util.concurrent.Callable;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.common.annotations.Beta;
import com.sap.cloud.sdk.cloudplatform.thread.Property;
import com.sap.cloud.sdk.cloudplatform.thread.ThreadContextAccessor;
import com.sap.cloud.sdk.cloudplatform.thread.ThreadContextDecorator;
import com.sap.cloud.security.token.SecurityContext;
import com.sap.cloud.security.token.Token;

import io.vavr.control.Try;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;

/**
 * Implementation of {@link ThreadContextDecorator} that ensures the correct initialization of {@link Token Tokens} when
 * working with non-container managed threads on Cloud Foundry.
 */
@Slf4j
@Beta
public class SecurityContextThreadContextDecorator implements ThreadContextDecorator
{
    @Getter
    private final int priority = DefaultPriorities.SCP_CF_SECURITY_CONTEXT_DECORATOR;

    @Nonnull
    @Override
    public <T> Callable<T> decorateCallable( @Nonnull final Callable<T> callable )
    {
        final Try<Token> tryToken =
            ThreadContextAccessor
                .getCurrentContext()
                .<Token> getProperty(SecurityContextThreadContextListener.PROPERTY_SECURITY_CONTEXT_TOKEN)
                .map(Property::getValue)
                .map(Objects::requireNonNull);

        if( tryToken.isFailure() ) {
            log.debug("Unable to pass token to SecurityContext of current thread.");
            return callable;
        }

        return () -> {
            @Nullable
            final Token previousToken = SecurityContext.getToken();

            log.debug("Passing token to SecurityContext.");
            SecurityContext.setToken(tryToken.get());

            final T result = callable.call();

            SecurityContext.setToken(previousToken);
            return result;
        };
    }
}
